﻿using System;

namespace TC_SeekThermal
{
    /// <summary>
    /// Description of ClassThermalFrame.
    /// </summary>
    public class ThermalFrame {
        public readonly int W = 0, H = 0;
        public ushort[,] Data;
        public ushort MinValue = 65535;
        public ushort MaxValue = 0;
        public ushort AvrValue = 0;
        public readonly byte StatusByte;
        public ushort DevTemp;
        public ushort FrameCnt;
        public byte[] RawData;
        bool _isSeekPro = false;
        int _loopW = 0;
        int _loopH = 0;

        internal ThermalFrame(byte[] data, int Width, int Height) {
            // Original data stream.
            if (data == null) { return; }
            RawData = data;
            W = Width; H = Height;
            _loopW = W - 1;
            _loopH = H - 1;
            if (W == 320) {
                _isSeekPro = true;
            }
            //RawDataBytes = data;
            //1368
            if (_isSeekPro) {
                StatusByte = data[4];
                DevTemp = (ushort)(data[11] << 8 | data[10]);
                FrameCnt = (ushort)(data[9] << 8 | data[8]);
            }
            else {
                StatusByte = data[20];
                DevTemp = (ushort)(data[3] << 8 | data[2]);
                FrameCnt = (ushort)(data[81] << 8 | data[80]);
            }

            //The first frames have the following IDs:
            //frame 1: 4	frame 5: 10
            //frame 2: 9	frame 6: 5
            //frame 3: 8	frame 7: 1
            //frame 4: 7	frame 8: 3
            //So, after the initialization sequence, you'll get something like this:
            //6, 1, 3, 3, 3, 6, 1, 3, 3, 3, 3, 3, 6, 1, 3, 3, 3, 3, 3, 3 etc.
            Data = new ushort[W, H];
            UInt32 AVR = 0; int cnt = 0; int x = 0, y = 0;
            //########## Seek Pro ############################################
            if (_isSeekPro) {
                for (int i = 2736; i < data.Length; i += 2) {
                    ushort val = (ushort)(data[i + 1] << 8 | data[i]);
                    if (x < W) {
                        if (val > 2000 && val < 12000) {
                            AVR += val; cnt++;
                            Data[x, y] = (ushort)val;
                        }
                    }
                    x++;
                    if (x == W + 22) {
                        y++;
                        x = 0;
                        if (y == H) {
                            break;
                        }
                    }
                }
                if (cnt > 0) {
                    AvrValue = (ushort)(AVR / cnt);
                }
                return;
            }
            //########## Seek Pro ############################################
            int HexagonCnt = 10;
            for (int i = 0; i < data.Length; i += 2) {
                ushort val = (ushort)(data[i + 1] << 8 | data[i]);
                if (i == HexagonCnt) {
                    HexagonCnt += 15;
                }
                else {
                    if (x < W) {
                        if (val > 2000 && val < 12000) {
                            AVR += val; cnt++;
                            Data[x, y] = (ushort)val;
                        }
                    }
                    else if (x == W) {
                        //Line Cal Pixel
                        //LineCal[y] = (ushort)((float)val/10f);
                    }
                }
                x++;
                if (x == W + 2) {
                    y++;
                    x = 0;
                    if (y == H) {
                        break;
                    }
                }
            }
            if (cnt > 0) {
                AvrValue = (ushort)(AVR / cnt);
            }
        }
        public bool[,] Cal_GetDefPixel() { //true = valid und false = defekt
            bool[,] output = new bool[W, H];
            for (int y = 0; y < H; ++y) {
                for (int x = 0; x < W; ++x) {
                    if ((Data[x, y] < 2000) || (Data[x, y] > 8000)) {
                        output[x, y] = false;
                    }
                    else {
                        if (_isSeekPro) {
                            output[x, y] = true;
                        }
                        else {
                            output[x, y] = !is_pattern_pixel(x, y);
                        }
                    }
                }
            }
            if (!_isSeekPro) {
                output[1, 0] = false; //TempValue
                output[40, 0] = false; //FrameCNT
            }
            return output;
        }
        public double[,] Cal_GetGains(ref bool[,] DPMap, ushort Teiler) {
            double[,] output = new double[W, H];
            for (int y = 0; y < H; ++y) {
                for (int x = 0; x < W; ++x) {
                    if (!DPMap[x, y]) { output[x, y] = 1f; continue; } //invalid ignorieren
                    float source = Data[x, y];
                    if (source > 2000 && source < 8000) {
                        double val = (double)(Teiler) / (double)(source);
                        if (val < -100 || val > 100) { val = 1f; DPMap[x, y] = false; }
                        if (val == 0) { val = 1f; DPMap[x, y] = false; }
                        output[x, y] = (double)val;
                    }
                    else {
                        DPMap[x, y] = false;
                        output[x, y] = 1f;
                    }
                }
            }
            return output;
        }
        public void Frame_ShutterGaincal(ref bool[,] DPMap, ushort[,] Shutter, double[,] GMap, ushort avr) {
            for (int y = 0; y < H; ++y) {
                for (int x = 0; x < W; ++x) {
                    if (!DPMap[x, y]) { continue; } //invalid ignorieren
                    int val = (int)Data[x, y];
                    val = (int)((double)(Data[x, y] - Shutter[x, y]) * GMap[x, y]) + avr;
                    if (val < 0) { val = 0; }
                    if (val > 0xffff) { val = 0xffff; }
                    Data[x, y] = (ushort)val;
                }
            }
        }
        public void Frame_RemoveDeathPixel(bool[,] DPMap) {
            //1  2  3
            //4  px 6
            //7  8  9
            int _lowW = _loopW - 1;
            int _lowH = _loopH - 1;
            //Mitte ####################
            for (int y = 1; y < _loopH; ++y) {
                for (int x = 1; x < _loopW; ++x) {
                    if (DPMap[x, y]) { continue; } //Pixel ist valid
                    long Val = 0; byte Cnt = 0;
                    if (DPMap[x - 1, y - 1]) { Val += Data[x - 1, y - 1]; Cnt++; } //1
                    if (DPMap[x, y - 1]) { Val += Data[x, y - 1]; Cnt++; } //2
                    if (DPMap[x + 1, y - 1]) { Val += Data[x + 1, y - 1]; Cnt++; } //3
                    if (DPMap[x - 1, y]) { Val += Data[x - 1, y]; Cnt++; } //4
                    if (DPMap[x + 1, y]) { Val += Data[x + 1, y]; Cnt++; } //6
                    if (DPMap[x - 1, y + 1]) { Val += Data[x - 1, y + 1]; Cnt++; } //7
                    if (DPMap[x, y + 1]) { Val += Data[x, y + 1]; Cnt++; } //8
                    if (DPMap[x + 1, y + 1]) { Val += Data[x + 1, y + 1]; Cnt++; } //9
                    if (Cnt != 0) { Data[x, y] = (ushort)(Val / Cnt); }
                }
            }
            //Rand Links ####################
            for (int y = 1; y < _loopH; ++y) {
                if (DPMap[0, y]) { continue; } //Pixel ist valid
                long Val = 0; byte Cnt = 0;
                if (DPMap[0, y - 1]) { Val += Data[0, y - 1]; Cnt++; } //2
                if (DPMap[1, y - 1]) { Val += Data[1, y - 1]; Cnt++; } //3
                if (DPMap[1, y]) { Val += Data[1, y]; Cnt++; } //6
                if (DPMap[0, y + 1]) { Val += Data[0, y + 1]; Cnt++; } //8
                if (DPMap[1, y + 1]) { Val += Data[1, y + 1]; Cnt++; } //9
                if (Cnt != 0) { Data[0, y] = (ushort)(Val / Cnt); }
            }
            //Rand Rechts ####################
            for (int y = 1; y < _loopH; ++y) {
                if (DPMap[_loopW, y]) { continue; } //Pixel ist valid
                long Val = 0; byte Cnt = 0;
                if (DPMap[_lowW, y - 1]) { Val += Data[_lowW, y - 1]; Cnt++; } //1
                if (DPMap[_loopW, y - 1]) { Val += Data[_loopW, y - 1]; Cnt++; } //2
                if (DPMap[_lowW, y]) { Val += Data[_lowW, y]; Cnt++; } //4
                if (DPMap[_lowW, y + 1]) { Val += Data[_lowW, y + 1]; Cnt++; } //7
                if (DPMap[_loopW, y + 1]) { Val += Data[_loopW, y + 1]; Cnt++; } //8
                if (Cnt != 0) { Data[_loopW, y] = (ushort)(Val / Cnt); }
            }
            //Rand Oben ####################
            for (int x = 1; x < _loopW; ++x) {
                if (DPMap[x, 0]) { continue; } //Pixel ist valid
                long Val = 0; byte Cnt = 0;
                if (DPMap[x - 1, 0]) { Val += Data[x - 1, 0]; Cnt++; } //4
                if (DPMap[x + 1, 0]) { Val += Data[x + 1, 0]; Cnt++; } //6
                if (DPMap[x - 1, 1]) { Val += Data[x - 1, 1]; Cnt++; } //7
                if (DPMap[x, 1]) { Val += Data[x, 1]; Cnt++; } //8
                if (DPMap[x + 1, 1]) { Val += Data[x + 1, 1]; Cnt++; } //9
                if (Cnt != 0) { Data[x, 0] = (ushort)(Val / Cnt); }
            }
            //Rand Unten ####################
            for (int x = 1; x < _loopW; ++x) {
                if (DPMap[x, _loopH]) { continue; } //Pixel ist valid
                long Val = 0; byte Cnt = 0;
                if (DPMap[x - 1, _lowH]) { Val += Data[x - 1, _lowH]; Cnt++; } //1
                if (DPMap[x, _lowH]) { Val += Data[x, _lowH]; Cnt++; } //2
                if (DPMap[x + 1, _lowH]) { Val += Data[x + 1, _lowH]; Cnt++; } //3
                if (DPMap[x - 1, _loopH]) { Val += Data[x - 1, _loopH]; Cnt++; } //4
                if (DPMap[x + 1, _loopH]) { Val += Data[x + 1, _loopH]; Cnt++; } //6
                if (Cnt != 0) { Data[x, _loopH] = (ushort)(Val / Cnt); }
            }
            //Eckenpixel ################
            //oben links
            if (!DPMap[0, 0]) {
                long Val = 0; byte Cnt = 0;
                if (DPMap[1, 0]) { Val += Data[1, 0]; Cnt++; } //6
                if (DPMap[0, 1]) { Val += Data[0, 1]; Cnt++; } //8
                if (DPMap[1, 1]) { Val += Data[1, 1]; Cnt++; } //9
                if (Cnt != 0) { Data[0, 0] = (ushort)(Val / Cnt); }
            }
            //oben rechts
            if (!DPMap[_loopW, 0]) {
                long Val = 0; byte Cnt = 0;
                if (DPMap[_lowW, 0]) { Val += Data[_lowW, 0]; Cnt++; } //4
                if (DPMap[_lowW, 1]) { Val += Data[_lowW, 1]; Cnt++; } //7
                if (DPMap[_loopW, 1]) { Val += Data[_loopW, 1]; Cnt++; } //8
                if (Cnt != 0) { Data[_loopW, 0] = (ushort)(Val / Cnt); }
            }
            //unten links
            if (!DPMap[0, _loopH]) {
                long Val = 0; byte Cnt = 0;
                if (DPMap[0, _lowH]) { Val += Data[0, _lowH]; Cnt++; } //2
                if (DPMap[1, _lowH]) { Val += Data[1, _lowH]; Cnt++; } //3
                if (DPMap[1, _loopH]) { Val += Data[1, _loopH]; Cnt++; } //6
                if (Cnt != 0) { Data[0, _loopH] = (ushort)(Val / Cnt); }
            }
            //unten rechts
            if (!DPMap[_loopW, _loopH]) {
                long Val = 0; byte Cnt = 0;
                if (DPMap[_lowW, _lowH]) { Val += Data[_lowW, _lowH]; Cnt++; } //1
                if (DPMap[_loopW, _lowH]) { Val += Data[_loopW, _lowH]; Cnt++; } //2
                if (DPMap[_lowW, _loopH]) { Val += Data[_lowW, _loopH]; Cnt++; } //4
                if (Cnt != 0) { Data[_loopW, _loopH] = (ushort)(Val / Cnt); }
            }
        }
        public void Frame_CalcMinMax(bool[,] DPMap, bool UseZeroOffset) {
            UInt32 AVR = 0; int cnt = 0;
            MinValue = 0xffff;
            MaxValue = 0;
            if (UseZeroOffset) {
                for (int y = 0; y < H; ++y) {
                    for (int x = 0; x < W; ++x) {
                        if (!DPMap[x, y]) { continue; } //invalid ignorieren
                        int data = Data[x, y];
                        if (data < 3000) {
                            data = MaxValue;
                            Data[x, y] = (ushort)data;
                        }
                        else {
                            AVR += Data[x, y]; cnt++;
                            if (Data[x, y] < MinValue) { MinValue = Data[x, y]; }
                            if (Data[x, y] > MaxValue) { MaxValue = Data[x, y]; }
                        }
                    }
                }
            }
            else {
                for (int y = 0; y < H; ++y) {
                    for (int x = 0; x < W; ++x) {
                        if (!DPMap[x, y]) { continue; } //invalid ignorieren
                        AVR += Data[x, y]; cnt++;
                        if (Data[x, y] < MinValue) { MinValue = Data[x, y]; }
                        if (Data[x, y] > MaxValue) { MaxValue = Data[x, y]; }
                    }
                }
            }

            if (cnt > 0) {
                AvrValue = (ushort)(AVR / (long)cnt);
            }
        }

        bool is_pattern_pixel(int x, int y) {
            int pattern_start = (10 - y * 4) % 15;
            if (pattern_start < 0) { pattern_start = 15 + pattern_start; }
            return (x >= pattern_start && ((x - pattern_start) % 15) == 0);
        }
    }
}
