﻿#region Usings
using System.ComponentModel;
using System.Runtime.InteropServices;
using Microsoft.Win32.SafeHandles;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Text.RegularExpressions;
#endregion

namespace WinUSB
{
	public static class WinUSBErr
	{
		public static bool HasErr=false;
    	public static string LastErr="";
	}
	#region NativeMethods
    
    class EnumeratedDevice
    {
        public string DevicePath { get; set; }
        public Guid InterfaceGuid { get; set; }
        public string DeviceDescription { get; set; }
        public string Manufacturer { get; set; }
        public string FriendlyName { get; set; }
    }
    static class NativeMethods
    {
        struct SP_DEVINFO_DATA
        {
            public UInt32 cbSize;
            public Guid classGuid;
            public UInt32 devInst;
            public IntPtr reserved;
        }
        struct SP_DEVICE_INTERFACE_DATA
        {
            public UInt32 cbSize;
            public Guid interfaceClassGuid;
            public UInt32 flags;
            public IntPtr reserved;
        }
        struct DEVPROPKEY
        {
            public Guid fmtId;
            public UInt32 pId;


            //DEFINE_DEVPROPKEY(DEVPKEY_Device_DeviceDesc,             0xa45c254e, 0xdf1c, 0x4efd, 0x80, 0x20, 0x67, 0xd1, 0x46, 0xa8, 0x50, 0xe0, 2);     // DEVPROP_TYPE_STRING
            //DEFINE_DEVPROPKEY(DEVPKEY_Device_Manufacturer,           0xa45c254e, 0xdf1c, 0x4efd, 0x80, 0x20, 0x67, 0xd1, 0x46, 0xa8, 0x50, 0xe0, 13);    // DEVPROP_TYPE_STRING
            //DEFINE_DEVPROPKEY(DEVPKEY_Device_FriendlyName,           0xa45c254e, 0xdf1c, 0x4efd, 0x80, 0x20, 0x67, 0xd1, 0x46, 0xa8, 0x50, 0xe0, 14);    // DEVPROP_TYPE_STRING

            public static DEVPROPKEY Device_DeviceDesc { get { return new DEVPROPKEY() { fmtId = new Guid("a45c254e-df1c-4efd-8020-67d146a850e0"), pId = 2 }; } }
            public static DEVPROPKEY Device_Manufacturer { get { return new DEVPROPKEY() { fmtId = new Guid("a45c254e-df1c-4efd-8020-67d146a850e0"), pId = 13 }; } }
            public static DEVPROPKEY Device_FriendlyName { get { return new DEVPROPKEY() { fmtId = new Guid("a45c254e-df1c-4efd-8020-67d146a850e0"), pId = 14 }; } }
        }
		public struct WINUSB_SETUP_PACKET
        {
            public byte RequestType;
            public byte Request;
            public UInt16 Value;
            public UInt16 Index;
            public UInt16 Length;
        }
        
        #region const...
        const UInt32 DIGCF_PRESENT = 2;
        const UInt32 DIGCF_ALLCLASSES = 4;
        const UInt32 DIGCF_DEVICEINTERFACE = 0x10;

        private static IntPtr INVALID_HANDLE_VALUE = new IntPtr(-1);

        const int ERROR_NOT_FOUND = 1168;
        const int ERROR_FILE_NOT_FOUND = 2;
        const int ERROR_NO_MORE_ITEMS = 259;
        const int ERROR_INSUFFICIENT_BUFFER = 122;
        const int ERROR_MORE_DATA = 234;

        public const int ERROR_SEM_TIMEOUT = 121;
        public const int ERROR_IO_PENDING = 997;

        const int DICS_FLAG_GLOBAL = 1;
        const int DICS_FLAG_CONFIGSPECIFIC = 2;

        const int DIREG_DEV = 1;
        const int DIREG_DRV = 2;

        const int KEY_READ = 0x20019; // Registry SAM value.

        const int RRF_RT_REG_SZ = 2;
        const int RRF_RT_REG_MULTI_SZ = 0x20;
        
        public const uint FILE_ATTRIBUTE_NORMAL = 0x80;
        public const uint FILE_FLAG_OVERLAPPED = 0x40000000;
        public const uint GENERIC_READ = 0x80000000;
        public const uint GENERIC_WRITE = 0x40000000;
        public const uint CREATE_NEW = 1;
        public const uint CREATE_ALWAYS = 2;
        public const uint OPEN_EXISTING = 3;
        public const uint FILE_SHARE_READ = 1;
        public const uint FILE_SHARE_WRITE = 2;
		#endregion

        /// <summary>
        /// Retrieve device paths that can be opened from a specific device interface guid.
        /// todo: Make this friendlier & query some more data about the devices being returned.
        /// </summary>
        /// <param name="deviceInterface">Guid uniquely identifying the interface to search for</param>
        /// <returns>List of device paths that can be opened with CreateFile</returns>
        public static EnumeratedDevice[] EnumerateDevicesByInterface(Guid deviceInterface)
        {
            // Horribe horrible things have to be done with SetupDI here. These travesties must never leave this class.
            List<EnumeratedDevice> outputPaths = new List<EnumeratedDevice>();

            IntPtr devInfo = SetupDiGetClassDevs(ref deviceInterface, null, IntPtr.Zero, DIGCF_DEVICEINTERFACE | DIGCF_PRESENT);
            if(devInfo == INVALID_HANDLE_VALUE)
            {
                throw new Exception("SetupDiGetClassDevs failed. " + (new Win32Exception()).ToString());
            }

            try
            {
                uint deviceIndex = 0;
                SP_DEVICE_INTERFACE_DATA interfaceData = new SP_DEVICE_INTERFACE_DATA();

                bool success = true;
                for (deviceIndex = 0; ; deviceIndex++)
                {
                    interfaceData.cbSize = (uint)Marshal.SizeOf(interfaceData);
                    success = SetupDiEnumDeviceInterfaces(devInfo, IntPtr.Zero, ref deviceInterface, deviceIndex, ref interfaceData);
                    if (!success)
                    {
                        if (Marshal.GetLastWin32Error() != ERROR_NO_MORE_ITEMS)
                        {
                            throw new Exception("SetupDiEnumDeviceInterfaces failed " + (new Win32Exception()).ToString());
                        }
                        // We have reached the end of the list of devices.
                        break;
                    }

                    // This is a valid interface, retrieve its path
                    EnumeratedDevice dev = new EnumeratedDevice() { DevicePath = RetrieveDeviceInstancePath(devInfo, interfaceData), InterfaceGuid = deviceInterface };


                    // Todo: debug. Not working correctly.
                    /*
                    dev.DeviceDescription = RetrieveDeviceInstancePropertyString(devInfo, interfaceData, DEVPROPKEY.Device_DeviceDesc);
                    dev.Manufacturer = RetrieveDeviceInstancePropertyString(devInfo, interfaceData, DEVPROPKEY.Device_Manufacturer);
                    dev.FriendlyName = RetrieveDeviceInstancePropertyString(devInfo, interfaceData, DEVPROPKEY.Device_FriendlyName);
                    */

                    outputPaths.Add(dev);
                }
            }
            finally
            {
                SetupDiDestroyDeviceInfoList(devInfo);
            }

            return outputPaths.ToArray();
        }
        public static EnumeratedDevice[] EnumerateAllWinUsbDevices()
        {
            List<EnumeratedDevice> outputDevices = new List<EnumeratedDevice>();
            string[] guids = EnumerateAllWinUsbGuids();
            foreach (string guid in guids)
            {
                try
                {
                    Guid g = new Guid(guid);
                    outputDevices.AddRange(EnumerateDevicesByInterface(g));
                }
                catch
                {
                    // Ignore failing guid conversions.
                }
            }
            return outputDevices.ToArray();
        }
        public static string[] EnumerateAllWinUsbGuids()
        {
            // Horribe horrible things have to be done with SetupDI here. These travesties must never leave this class.
            List<string> outputGuids = new List<string>();

            IntPtr devInfo = SetupDiGetClassDevs(IntPtr.Zero, null, IntPtr.Zero, DIGCF_ALLCLASSES | DIGCF_PRESENT);
            if (devInfo == INVALID_HANDLE_VALUE) {
                throw new Exception("SetupDiGetClassDevs failed. " + (new Win32Exception()).ToString());
            }

            try {
                uint deviceIndex = 0;
                SP_DEVINFO_DATA devInfoData = new SP_DEVINFO_DATA();

                bool success = true;
                for (deviceIndex = 0; ; deviceIndex++) {
                    devInfoData.cbSize = (uint)Marshal.SizeOf(devInfoData);
                    success = SetupDiEnumDeviceInfo(devInfo, deviceIndex, ref devInfoData);
                    if (!success) {
                        if (Marshal.GetLastWin32Error() != ERROR_NO_MORE_ITEMS) {
                            throw new Exception("SetupDiEnumDeviceInfo failed " + (new Win32Exception()).ToString());
                        }
                        // We have reached the end of the list of devices.
                        break;
                    }

                    // Enumerate the WinUSB Interface Guids (if present) by looking at the registry.
                    //DebugEnumRegistryValues(devInfo, devInfoData);
                    string guid = RetrieveDeviceProperty(devInfo, devInfoData, "DeviceInterfaceGUIDs");
                    if(guid==null||guid.EndsWith("!")) {
                        guid = RetrieveDeviceProperty(devInfo, devInfoData, "DeviceInterfaceGUID");
                    }

                    if (guid!=null&&!guid.EndsWith("!")) {
                        outputGuids.Add(guid);
                    }
                }
            }
            finally
            {
                SetupDiDestroyDeviceInfoList(devInfo);
            }

            return outputGuids.Distinct().ToArray();
        }

        static void DebugEnumRegistryValues(IntPtr devInfo, SP_DEVINFO_DATA devInfoData)
        {
            System.Console.WriteLine("DebugEnumRegistryValues");
            IntPtr hKey = SetupDiOpenDevRegKey(devInfo, ref devInfoData, DICS_FLAG_GLOBAL, 0, DIREG_DEV, KEY_READ);
            if (hKey == INVALID_HANDLE_VALUE)
            {
                System.Console.WriteLine("Failed");
                return;
                throw new Exception("SetupDiGetClassDevs failed. " + (new Win32Exception()).ToString());
            }

            try
            {

                IntPtr memValue = Marshal.AllocHGlobal(16384);
                try
                {
                    IntPtr memData = Marshal.AllocHGlobal(65536);
                    try
                    {

                        for (int i = 0; ; i++)
                        {
                            UInt32 outValueLen = 16384;
                            UInt32 outDataLen = 65536;
                            UInt32 outType;
                            long output = RegEnumValue(hKey, (uint)i, memValue, ref outValueLen, IntPtr.Zero, out outType, memData, ref outDataLen);
                            if((int)output == ERROR_NO_MORE_ITEMS)
                            {
                                break;
                            }
                            if (output != 0)
                            {
                                throw new Exception("RegEnumValue failed " + (new Win32Exception((int)output)).ToString());
                            }

                            string value = ReadAsciiString(memValue, (int)outValueLen,0);
                            string data = ReadAsciiString(memData, (int)outDataLen,0);

                            Console.WriteLine("Enum: '{0}' -  {2} '{1}'", value, data, outType);

                        }

                    }
                    finally
                    {
                        Marshal.FreeHGlobal(memData);
                    }
                }
                finally
                {
                    Marshal.FreeHGlobal(memValue);
                }

            }
            finally
            {
                RegCloseKey(hKey);
            }
        }
        static string RetrieveDeviceProperty(IntPtr devInfo, SP_DEVINFO_DATA devInfoData, string deviceProperty)
        {
            IntPtr hKey = SetupDiOpenDevRegKey(devInfo, ref devInfoData, DICS_FLAG_GLOBAL, 0, DIREG_DEV, KEY_READ);
            if (hKey == INVALID_HANDLE_VALUE) {
                return null; // Ignore failures, probably the key doesn't exist.
            }

            try {
                UInt32 outType;
                UInt32 outLength = 0;
                //static extern int RegQueryValueEx( UIntPtr hKey, string lpValueName, int lpReserved, out uint lpType, System.Text.StringBuilder lpData, ref uint lpcbData);
                //long output = RegGetValueA(hKey,null, deviceProperty, RRF_RT_REG_SZ | RRF_RT_REG_MULTI_SZ, out outType, IntPtr.Zero, ref outLength);
                long output = RegQueryValueEx(hKey,deviceProperty,0,out outType, IntPtr.Zero,out outLength);
                if(output == ERROR_FILE_NOT_FOUND) { //output==2
                    return null; // Key not present, don't continue.
                }
                
                if (output != 0) {
                	return "RegGetValue failed (determining length): "+output.ToString()+" !";
                }
                //output==0
                //deviceProperty==DeviceInterfaceGUIDs
                //outType==7
                //outLength=42
                IntPtr mem = Marshal.AllocHGlobal((int)outLength);
                try {

                    UInt32 actualLength = outLength;
                    //output = RegGetValueA(hKey, null, deviceProperty, RRF_RT_REG_SZ | RRF_RT_REG_MULTI_SZ, out outType, mem, ref actualLength);
	                output = RegQueryValueEx(hKey,deviceProperty,0,out outType, mem,out actualLength);
                    //erster run: outType=7 mem=6786192,12011880 actualLength=40
	                //zweiter run: outType=7 mem=7026584,12275432 actualLength=40
                    if (output != 0) {
                    	return "RegGetValue failed (retrieving data): "+output.ToString()+" !";
                    }

                    // Convert TCHAR string into chars.
                    if (actualLength > outLength) {
                    	return "Consistency issue: Actual length should not be larger than buffer size. !";
                    }

                    return ReadAsciiString(mem, (int)((actualLength)),0);
                } finally {
                    Marshal.FreeHGlobal(mem);
                }
            } catch (Exception err){
            	WinUSBErr.LastErr="Error in RetrieveDeviceProperty: "+err.Message;
        		WinUSBErr.HasErr=true;
            } finally {
                RegCloseKey(hKey);
            }
            return null;
        }
        static string RetrieveDeviceInstancePath(IntPtr devInfo, SP_DEVICE_INTERFACE_DATA interfaceData)
        {
            // This is a valid interface, retrieve its path
            UInt32 requiredLength = 0;

            if (!SetupDiGetDeviceInterfaceDetail(devInfo, ref interfaceData, IntPtr.Zero, 0, ref requiredLength, IntPtr.Zero)) {
                int err = Marshal.GetLastWin32Error();

                if (err != ERROR_INSUFFICIENT_BUFFER) {
                    throw new Exception("SetupDiGetDeviceInterfaceDetail failed (determining length) " + (new Win32Exception()).ToString());
                }
            }

            UInt32 actualLength = requiredLength;
            Int32 structLen = 6;
            if (IntPtr.Size == 8) structLen = 8; // Compensate for 64bit struct packing.

            if (requiredLength < structLen) {
                throw new Exception("Consistency issue: Required memory size should be larger");
            }

            IntPtr mem = Marshal.AllocHGlobal((int)requiredLength);
            
            try
            {
                Marshal.WriteInt32(mem, structLen); // set fake size in fake structure

                if (!SetupDiGetDeviceInterfaceDetail(devInfo, ref interfaceData, mem, requiredLength, ref actualLength, IntPtr.Zero))
                {
                    throw new Exception("SetupDiGetDeviceInterfaceDetail failed (retrieving data) " + (new Win32Exception()).ToString());
                }

                // Convert TCHAR string into chars.
                if (actualLength > requiredLength)
                {
                    throw new Exception("Consistency issue: Actual length should not be larger than buffer size.");
                }

                return ReadString(mem, (int)((actualLength - 4) / 2), 4);
            }
            finally
            {
                Marshal.FreeHGlobal(mem);
            }
        }
        static string RetrieveDeviceInstancePropertyString(IntPtr devInfo, SP_DEVICE_INTERFACE_DATA interfaceData, DEVPROPKEY property)
        {
            // This is a valid interface, retrieve its path
            UInt32 requiredLength = 0;
            UInt32 propertyType;

            if (!SetupDiGetDeviceInterfaceProperty(devInfo, ref interfaceData, ref property, out propertyType, IntPtr.Zero, 0, out requiredLength, 0))
            {
                int err = Marshal.GetLastWin32Error();
                if (err == ERROR_NOT_FOUND)
                {
                    return null;
                }

                if (err != ERROR_INSUFFICIENT_BUFFER)
                {
                    throw new Exception("SetupDiGetDeviceInterfaceProperty failed (determining length) " + (new Win32Exception()).ToString());
                }

            }

            UInt32 actualLength = requiredLength;


            IntPtr mem = Marshal.AllocHGlobal((int)requiredLength);
            try
            {
                Marshal.WriteInt32(mem, 6); // set fake size in fake structure

                if (!SetupDiGetDeviceInterfaceProperty(devInfo, ref interfaceData, ref property, out propertyType, mem, requiredLength, out actualLength, 0))
                {
                    throw new Exception("SetupDiGetDeviceInterfaceProperty failed (retrieving data) " + (new Win32Exception()).ToString());
                }

                // Convert TCHAR string into chars.
                if (actualLength > requiredLength)
                {
                    throw new Exception("Consistency issue: Actual length should not be larger than buffer size.");
                }

                return ReadString(mem, (int)((actualLength) / 2),0);
            }
            finally
            {
                Marshal.FreeHGlobal(mem);
            }
        }
        static string ReadString(IntPtr source, int length, int offset)
        {//IntPtr source, int length, int offset = 0
            char[] stringChars = new char[length];
            for (int i = 0; i < length; i++)
            {
                stringChars[i] = (char)Marshal.ReadInt16(source, i * 2 + offset);
                if (stringChars[i] == 0) { length = i; break; }
            }
            return new string(stringChars, 0, length);
        }
        static string ReadAsciiString(IntPtr source, int length, int offset)
        {//IntPtr source, int length, int offset = 0
            char[] stringChars = new char[length];
            for (int i = 0; i < length; i++)
            {
                stringChars[i] = (char)Marshal.ReadByte(source, i + offset);
                if (stringChars[i] == 0) { length = i; break; }
            }
            return new string(stringChars, 0, length);
        }
	
        #region DLL_imports
        /*
        HDEVINFO SetupDiGetClassDevs(
          _In_opt_  const GUID *ClassGuid,
          _In_opt_  PCTSTR Enumerator,
          _In_opt_  HWND hwndParent,
          _In_      DWORD Flags
        );
         */
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static IntPtr SetupDiGetClassDevs(ref Guid classGuid, string enumerator, IntPtr hwndParent, UInt32 flags);
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static IntPtr SetupDiGetClassDevs(IntPtr classGuid, string enumerator, IntPtr hwndParent, UInt32 flags);

        /*
         BOOL SetupDiEnumDeviceInterfaces(
          _In_      HDEVINFO DeviceInfoSet,
          _In_opt_  PSP_DEVINFO_DATA DeviceInfoData,
          _In_      const GUID *InterfaceClassGuid,
          _In_      DWORD MemberIndex,
          _Out_     PSP_DEVICE_INTERFACE_DATA DeviceInterfaceData
        );
         */
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, IntPtr optDeviceInfoData, ref Guid interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, IntPtr optDeviceInfoData, IntPtr interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, ref SP_DEVINFO_DATA deviceInfoData, ref Guid interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static bool SetupDiEnumDeviceInterfaces(IntPtr deviceInfoSet, ref SP_DEVINFO_DATA deviceInfoData, IntPtr interfaceClassGuid, UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);


        /*
        BOOL SetupDiEnumDeviceInfo(
          _In_   HDEVINFO DeviceInfoSet,
          _In_   DWORD MemberIndex,
          _Out_  PSP_DEVINFO_DATA DeviceInfoData
        );
         */
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static bool SetupDiEnumDeviceInfo(IntPtr deviceInfoSet, UInt32 memberIndex, ref SP_DEVINFO_DATA deviceInfoData);

        /*
        HKEY SetupDiOpenDevRegKey(
          _In_  HDEVINFO DeviceInfoSet,
          _In_  PSP_DEVINFO_DATA DeviceInfoData,
          _In_  DWORD Scope,
          _In_  DWORD HwProfile,
          _In_  DWORD KeyType,
          _In_  REGSAM samDesired
        );
        */
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static IntPtr SetupDiOpenDevRegKey(IntPtr deviceInfoSet, ref SP_DEVINFO_DATA deviceInfoData, UInt32 scope, UInt32 hwProfile, UInt32 keyType, UInt32 samDesired);

        /*
        BOOL SetupDiGetDeviceInterfaceProperty(
          _In_       HDEVINFO DeviceInfoSet,
          _In_       PSP_DEVICE_INTERFACE_DATA DeviceInterfaceData,
          _In_       const DEVPROPKEY *PropertyKey,
          _Out_      DEVPROPTYPE *PropertyType,
          _Out_      PBYTE PropertyBuffer,
          _In_       DWORD PropertyBufferSize,
          _Out_opt_  PDWORD RequiredSize,
          _In_       DWORD Flags
        );
        */
        [DllImport("setupapi.dll", SetLastError = true, CharSet=CharSet.Unicode, EntryPoint="SetupDiGetDeviceInterfacePropertyW")]
        extern static bool SetupDiGetDeviceInterfaceProperty(IntPtr deviceInfoSet, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceDataa, ref DEVPROPKEY propertyKey, out UInt32 propertyType, IntPtr outPropertyData, UInt32 dataBufferLength, out UInt32 requredBufferLength, UInt32 flags);


        /*
        BOOL SetupDiGetDevicePropertyKeys(
          _In_       HDEVINFO DeviceInfoSet,
          _In_       PSP_DEVINFO_DATA DeviceInfoData,
          _Out_opt_  DEVPROPKEY *PropertyKeyArray,
          _In_       DWORD PropertyKeyCount,
          _Out_opt_  PDWORD RequiredPropertyKeyCount,
          _In_       DWORD Flags
        );
        */

        /*
        LONG WINAPI RegCloseKey(
        _In_  HKEY hKey
        );
        */
        [DllImport("advapi32.dll", SetLastError = false)]
        extern static int RegCloseKey(IntPtr hKey);


        /*
        LONG WINAPI RegGetValue(
          _In_         HKEY hkey,
          _In_opt_     LPCTSTR lpSubKey,
          _In_opt_     LPCTSTR lpValue,
          _In_opt_     DWORD dwFlags,
          _Out_opt_    LPDWORD pdwType,
          _Out_opt_    PVOID pvData,
          _Inout_opt_  LPDWORD pcbData
        );
        */
        [DllImport("advapi32.dll", SetLastError = false)]//RegGetValue
        extern static int RegGetValueA(IntPtr hKey, string lpSubKey, string lpValue, UInt32 flags, out UInt32 outType, IntPtr outData, ref UInt32 dataLength);
//		
        [DllImport("advapi32.dll",EntryPoint="RegQueryValueEx")]
		public static extern int RegQueryValueEx(IntPtr hKey,string lpValueName,int lpReserved,out uint lpType, IntPtr lpData,out uint lpcbData);

        //[DllImport("advapi32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
//		static extern int RegGetValue( IntPtr hKey, string lpValueName, int lpReserved, out uint lpType, System.Text.StringBuilder lpData, ref uint lpcbData);
//        /*
//        LONG WINAPI RegEnumValue(
//          _In_         HKEY hKey,
//          _In_         DWORD dwIndex,
//          _Out_        LPTSTR lpValueName,
//          _Inout_      LPDWORD lpcchValueName,
//          _Reserved_   LPDWORD lpReserved,
//          _Out_opt_    LPDWORD lpType,
//          _Out_opt_    LPBYTE lpData,
//          _Inout_opt_  LPDWORD lpcbData
//        );
//        */
        [DllImport("advapi32.dll", SetLastError = false)]
        extern static int RegEnumValue(IntPtr hKey, UInt32 index, IntPtr outValue, ref UInt32 valueLen, IntPtr reserved, out UInt32 outType, IntPtr outData, ref UInt32 dataLength);

		
 
 /*
        BOOL SetupDiDestroyDeviceInfoList(
          _In_  HDEVINFO DeviceInfoSet
        );
          */
        [DllImport("setupapi.dll", SetLastError = true)]
        extern static bool SetupDiDestroyDeviceInfoList(IntPtr deviceInfoSet);


        /* 
        BOOL SetupDiGetDeviceInterfaceDetail(
          _In_       HDEVINFO DeviceInfoSet,
          _In_       PSP_DEVICE_INTERFACE_DATA DeviceInterfaceData,
          _Out_opt_  PSP_DEVICE_INTERFACE_DETAIL_DATA DeviceInterfaceDetailData,
          _In_       DWORD DeviceInterfaceDetailDataSize,
          _Out_opt_  PDWORD RequiredSize,
          _Out_opt_  PSP_DEVINFO_DATA DeviceInfoData
        );
          */
        [DllImport("setupapi.dll", SetLastError = true, CharSet=CharSet.Unicode)]
        extern static bool SetupDiGetDeviceInterfaceDetail(IntPtr deviceInfoSet, [In] ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData, 
            IntPtr deviceInterfaceDetailData, UInt32 deviceInterfaceDetailSize, ref UInt32 requiredSize, IntPtr deviceInfoData );


        /* 
        HANDLE WINAPI CreateFile(
          _In_      LPCTSTR lpFileName,
          _In_      DWORD dwDesiredAccess,
          _In_      DWORD dwShareMode,
          _In_opt_  LPSECURITY_ATTRIBUTES lpSecurityAttributes,
          _In_      DWORD dwCreationDisposition,
          _In_      DWORD dwFlagsAndAttributes,
          _In_opt_  HANDLE hTemplateFile
        );
          */
        [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        public extern static SafeFileHandle CreateFile(string lpFileName, UInt32 dwDesiredAccess, 
            UInt32 dwShareMode, IntPtr lpSecurityAttributes, UInt32 dwCreationDisposition, UInt32 dwFlagsAndAttributes, IntPtr hTemplateFile);
		/* 
        BOOL __stdcall WinUsb_Initialize(
          _In_   HANDLE DeviceHandle,
          _Out_  PWINUSB_INTERFACE_HANDLE InterfaceHandle
        );
          */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_Initialize(SafeFileHandle deviceHandle, out IntPtr interfaceHandle);

        /* 
        BOOL __stdcall WinUsb_Free(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_Free(IntPtr interfaceHandle);

        /*
         BOOL __stdcall WinUsb_ControlTransfer(
          _In_       WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_       WINUSB_SETUP_PACKET SetupPacket,
          _Out_      PUCHAR Buffer,
          _In_       ULONG BufferLength,
          _Out_opt_  PULONG LengthTransferred,
          _In_opt_   LPOVERLAPPED Overlapped
        );
        */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_ControlTransfer(IntPtr interfaceHandle, WINUSB_SETUP_PACKET setupPacket, byte[] buffer, uint bufferLength, out UInt32 lengthTransferred, IntPtr overlapped);

        /* 
        BOOL __stdcall WinUsb_ReadPipe(
          _In_       WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_       UCHAR PipeID,
          _Out_      PUCHAR Buffer,
          _In_       ULONG BufferLength,
          _Out_opt_  PULONG LengthTransferred,
          _In_opt_   LPOVERLAPPED Overlapped
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_ReadPipe(IntPtr interfaceHandle, byte pipeId, IntPtr buffer, uint bufferLength, IntPtr lengthTransferred, IntPtr overlapped);
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_ReadPipe(IntPtr interfaceHandle, byte pipeId, [Out] byte[] buffer, uint bufferLength, ref UInt32 lengthTransferred, IntPtr overlapped);

        /* 
        BOOL __stdcall WinUsb_WritePipe(
          _In_       WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_       UCHAR PipeID,
          _In_       PUCHAR Buffer,
          _In_       ULONG BufferLength,
          _Out_opt_  PULONG LengthTransferred,
          _In_opt_   LPOVERLAPPED Overlapped
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_WritePipe(IntPtr interfaceHandle, byte pipeId, [In] byte[] buffer, uint bufferLength, IntPtr lengthTransferred, ref NativeOverlapped overlapped);
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_WritePipe(IntPtr interfaceHandle, byte pipeId, [In] byte[] buffer, uint bufferLength, ref UInt32 lengthTransferred, IntPtr overlapped);


        /* 
        BOOL __stdcall WinUsb_GetOverlappedResult(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_   LPOVERLAPPED lpOverlapped,
          _Out_  LPDWORD lpNumberOfBytesTransferred,
          _In_   BOOL bWait
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_GetOverlappedResult(IntPtr interfaceHandle, IntPtr overlapped, out UInt32 numberOfBytesTransferred, bool wait);


        /* 
        BOOL __stdcall WinUsb_SetPipePolicy(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_  UCHAR PipeID,
          _In_  ULONG PolicyType,
          _In_  ULONG ValueLength,
          _In_  PVOID Value
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_SetPipePolicy(IntPtr interfaceHandle, byte pipeId, UInt32 policyType, UInt32 valueLength, UInt32[] value);

        /* 
        BOOL __stdcall WinUsb_GetPipePolicy(
          _In_     WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_     UCHAR PipeID,
          _In_     ULONG PolicyType,
          _Inout_  PULONG ValueLength,
          _Out_    PVOID Value
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_GetPipePolicy(IntPtr interfaceHandle, byte pipeId, UInt32 policyType, ref UInt32 valueLength, UInt32[] value);


        /* 
        BOOL __stdcall WinUsb_FlushPipe(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_  UCHAR PipeID
        );
        */

        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_FlushPipe(IntPtr interfaceHandle, byte pipeId);


        /*
        BOOL __stdcall WinUsb_GetDescriptor(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_   UCHAR DescriptorType,
          _In_   UCHAR Index,
          _In_   USHORT LanguageID,
          _Out_  PUCHAR Buffer,
          _In_   ULONG BufferLength,
          _Out_  PULONG LengthTransferred
        );
        */


        /*
        BOOL __stdcall WinUsb_QueryInterfaceSettings(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_   UCHAR AlternateSettingNumber,
          _Out_  PUSB_INTERFACE_DESCRIPTOR UsbAltInterfaceDescriptor
        );
        */

        /*
        BOOL __stdcall WinUsb_GetCurrentAlternateSetting(
          _In_   WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _Out_  PUCHAR AlternateSetting
        );
         */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_GetCurrentAlternateSetting(IntPtr interfaceHandle, out byte alternateSetting);


        /*
        BOOL __stdcall WinUsb_SetCurrentAlternateSetting(
          _In_  WINUSB_INTERFACE_HANDLE InterfaceHandle,
          _In_  UCHAR AlternateSetting
        );
        */
        [DllImport("winusb.dll", SetLastError = true)]
        public extern static bool WinUsb_SetCurrentAlternateSetting(IntPtr interfaceHandle, byte alternateSetting);
        #endregion

    }
    public struct NativeOverlapped
    {
        public IntPtr Internal;
        public IntPtr InternalHigh;
        public long Pointer; // On 32bit systems this is 32bit, but it's merged with an "Offset" field which is 64bit.
        public IntPtr Event;
    }
    public class Overlapped : IDisposable
    { 
    	public ManualResetEvent WaitEvent;
        public NativeOverlapped OverlappedStructShadow;
        public IntPtr OverlappedStruct;
        
        public Overlapped()
        {
            WaitEvent = new ManualResetEvent(false);
            OverlappedStructShadow = new NativeOverlapped();
            OverlappedStructShadow.Event = WaitEvent.SafeWaitHandle.DangerousGetHandle();

            OverlappedStruct = Marshal.AllocHGlobal(Marshal.SizeOf(OverlappedStructShadow));
            Marshal.StructureToPtr(OverlappedStructShadow, OverlappedStruct, false);
        }
        public void Dispose()
        {
            Marshal.FreeCoTaskMem(OverlappedStruct);
            
            WaitEvent.Close();
            //WaitEvent.Dispose(); // Deaktiviert wegen sicherheitsebene
            GC.SuppressFinalize(this);
        }
    }
    public enum WinUsbPipePolicy
    {
        SHORT_PACKET_TERMINATE = 1,
        AUTO_CLEAR_STALL = 2,
        PIPE_TRANSFER_TIMEOUT = 3,
        IGNORE_SHORT_PACKETS = 4,
        ALLOW_PARTIAL_READS = 5,
        AUTO_FLUSH = 6,
        RAW_IO = 7,
        MAXIMUM_TRANSFER_SIZE = 8,
        RESET_PIPE_ON_RESUME = 9
    }
    #endregion
    
    #region WinUSBDevice
    public class WinUSBEnumeratedDevice
    {
        internal string DevicePath;
        internal EnumeratedDevice EnumeratedData;
        internal WinUSBEnumeratedDevice(EnumeratedDevice enumDev)
        {
            DevicePath = enumDev.DevicePath;
            EnumeratedData = enumDev;
            Match m = Regex.Match(DevicePath, @"vid_([\da-f]{4})");
            if (m.Success) { VendorID = Convert.ToUInt16(m.Groups[1].Value, 16); }
            m = Regex.Match(DevicePath, @"pid_([\da-f]{4})");
            if (m.Success) { ProductID = Convert.ToUInt16(m.Groups[1].Value, 16); }
            m = Regex.Match(DevicePath, @"mi_([\da-f]{2})");
            if (m.Success) { UsbInterface = Convert.ToByte(m.Groups[1].Value, 16); }
        }

        public string Path { get { return DevicePath; } }
        public UInt16 VendorID { get; private set; }
        public UInt16 ProductID { get; private set; }
        public Byte UsbInterface { get; private set; }
        public Guid InterfaceGuid { get { return EnumeratedData.InterfaceGuid; } }


        public override string ToString()
        {
            return string.Format("WinUSBEnumeratedDevice({0},{1})", DevicePath, InterfaceGuid);
        }
    }
    public class WinUSBDevice : IDisposable
    {
        public static IEnumerable<WinUSBEnumeratedDevice> EnumerateDevices(Guid deviceInterfaceGuid)
        {
            foreach (EnumeratedDevice devicePath in NativeMethods.EnumerateDevicesByInterface(deviceInterfaceGuid))
            {
                yield return new WinUSBEnumeratedDevice(devicePath);
            }
        }

        public static IEnumerable<WinUSBEnumeratedDevice> EnumerateAllDevices()
        {
            foreach (EnumeratedDevice devicePath in NativeMethods.EnumerateAllWinUsbDevices())
            {
                yield return new WinUSBEnumeratedDevice(devicePath);
            }
        }
        public delegate void NewDataCallback();

        string myDevicePath;
        SafeFileHandle deviceHandle;
        IntPtr WinusbHandle;

        internal bool Stopping = false;

        public WinUSBDevice(WinUSBEnumeratedDevice deviceInfo)
        {
        	if (deviceInfo==null) {
        		WinUSBErr.HasErr=true;
        		WinUSBErr.LastErr="deviceInfo==null";
        		return; 
        	}
            myDevicePath = deviceInfo.DevicePath;

            deviceHandle = NativeMethods.CreateFile(myDevicePath, NativeMethods.GENERIC_READ | NativeMethods.GENERIC_WRITE,
                NativeMethods.FILE_SHARE_READ | NativeMethods.FILE_SHARE_WRITE, IntPtr.Zero, NativeMethods.OPEN_EXISTING,
                NativeMethods.FILE_ATTRIBUTE_NORMAL | NativeMethods.FILE_FLAG_OVERLAPPED, IntPtr.Zero);

            if (deviceHandle.IsInvalid) {
            	WinUSBErr.LastErr="Winusb deviceHandle is invalid.";
        		WinUSBErr.HasErr=true;
            	return;
                //throw new Exception("Could not create file. " + (new Win32Exception()).ToString());
            }

            if (!NativeMethods.WinUsb_Initialize(deviceHandle, out WinusbHandle)) {
                WinusbHandle = IntPtr.Zero;
                WinUSBErr.LastErr="Could not Initialize WinUSB.";
        		WinUSBErr.HasErr=true;
                return;
//                throw new Exception("Could not Initialize WinUSB. " + (new Win32Exception()).ToString());
            }


        }

        public byte AlternateSetting
        {
            get {
                byte alt;
                if (!NativeMethods.WinUsb_GetCurrentAlternateSetting(WinusbHandle, out alt))
                {
                    throw new Exception("GetCurrentAlternateSetting failed. " + (new Win32Exception()).ToString());
                }
                return alt;
            } set {
                if (!NativeMethods.WinUsb_SetCurrentAlternateSetting(WinusbHandle, value))
                {
                    throw new Exception("SetCurrentAlternateSetting failed. " + (new Win32Exception()).ToString());
                }
            }
        }

        public void Dispose()
        {
            Stopping = true;
            if (deviceHandle==null) { return; }
            // Close handles which will cause background theads to stop working & exit.
            if (WinusbHandle != IntPtr.Zero)
            {
                NativeMethods.WinUsb_Free(WinusbHandle);
                WinusbHandle = IntPtr.Zero;
            }
            deviceHandle.Close();

            // Wait for pipe threads to quit
            foreach (BufferedPipeThread th in bufferedPipes.Values)
            {
                while (!th.Stopped) Thread.Sleep(5);
            }

            GC.SuppressFinalize(this);
        }

        public void Close()
        {
            Dispose();
        }

        public void FlushPipe(byte pipeId)
        {
            if (!NativeMethods.WinUsb_FlushPipe(WinusbHandle, pipeId))
            {
                throw new Exception("FlushPipe failed. " + (new Win32Exception()).ToString());
            }
        }

        public UInt32 GetPipePolicy(byte pipeId, WinUsbPipePolicy policyType)
        {
            UInt32[] data = new UInt32[1];
            UInt32 length = 4;

            if (!NativeMethods.WinUsb_GetPipePolicy(WinusbHandle, pipeId, (uint)policyType, ref length, data))
            {
                throw new Exception("GetPipePolicy failed. " + (new Win32Exception()).ToString());
            }

            return data[0];
        }

        public void SetPipePolicy(byte pipeId, WinUsbPipePolicy policyType, UInt32 newValue)
        {
            UInt32[] data = new UInt32[1];
            UInt32 length = 4;
            data[0] = newValue;

            if (!NativeMethods.WinUsb_SetPipePolicy(WinusbHandle, pipeId, (uint)policyType, length, data))
            {
                throw new Exception("SetPipePolicy failed. " + (new Win32Exception()).ToString());
            }
        }

        Dictionary<byte, BufferedPipeThread> bufferedPipes = new Dictionary<byte, BufferedPipeThread>();
        public void EnableBufferedRead(byte pipeId, int bufferCount, int multiPacketCount)
        {//byte pipeId, int bufferCount = 4, int multiPacketCount = 1
            if (!bufferedPipes.ContainsKey(pipeId))
            {
                bufferedPipes.Add(pipeId, new BufferedPipeThread(this, pipeId, bufferCount, multiPacketCount));
            }
        }

        public void StopBufferedRead(byte pipeId)
        {

        }

        public void BufferedReadNotifyPipe(byte pipeId, NewDataCallback callback)
        {
            if (!bufferedPipes.ContainsKey(pipeId))
            {
                throw new Exception("Pipe not enabled for buffered reads!");
            }
            bufferedPipes[pipeId].NewDataEvent += callback;
        }

        BufferedPipeThread GetInterface(byte pipeId, bool packetInterface)
        {
            if (!bufferedPipes.ContainsKey(pipeId))
            {
                throw new Exception("Pipe not enabled for buffered reads!");
            }
            BufferedPipeThread th = bufferedPipes[pipeId];
            if (!th.InterfaceBound)
            {
                th.InterfaceBound = true;
                th.PacketInterface = packetInterface;
            }
            else
            {
                if (th.PacketInterface != packetInterface)
                {
                    string message = string.Format("Pipe is already bound as a {0} interface - cannot bind to both Packet and Byte interfaces",
                                                   packetInterface ? "Byte" : "Packet");
                    throw new Exception(message);
                }
            }
            return th;
        }
        public IPipeByteReader BufferedGetByteInterface(byte pipeId)
        {
            return GetInterface(pipeId, false);
        }

        public IPipePacketReader BufferedGetPacketInterface(byte pipeId)
        {
            return GetInterface(pipeId, true);
        }



        public byte[] BufferedReadPipe(byte pipeId, int byteCount)
        {
            return BufferedGetByteInterface(pipeId).ReceiveBytes(byteCount);
        }

        public byte[] BufferedPeekPipe(byte pipeId, int byteCount)
        {
            return BufferedGetByteInterface(pipeId).PeekBytes(byteCount);
        }

        public void BufferedSkipBytesPipe(byte pipeId, int byteCount)
        {
            BufferedGetByteInterface(pipeId).SkipBytes(byteCount);
        }

        public byte[] BufferedReadExactPipe(byte pipeId, int byteCount)
        {
            return BufferedGetByteInterface(pipeId).ReceiveExactBytes(byteCount);
        }

        public int BufferedByteCountPipe(byte pipeId)
        {
            return BufferedGetByteInterface(pipeId).QueuedDataLength;
        }

        public UInt16[] ReadExactPipeU16(byte pipeId, int count)
        {
            int read = 0;
            UInt16[] accumulate = null;
            while (read < count)
            {
                UInt16[] data = ReadPipeU16(pipeId, count - read);
                if (data.Length == 0)
                {
                    // Timeout happened in ReadPipeU16.
                    throw new Exception("Timed out while trying to read data.");
                }
                if (data.Length == count) return data;
                if (accumulate == null)
                {
                    accumulate = new UInt16[count];
                }
                Array.Copy(data, 0, accumulate, read, data.Length);
                read += data.Length;
            }
            return accumulate;
        }

        public byte[] ReadExactPipe(byte pipeId, int byteCount)
        {
            int read = 0;
            byte[] accumulate = null;
            while (read < byteCount)
            {
                byte[] data = ReadPipe(pipeId, byteCount - read);
                if (data.Length == 0)
                {
                    // Timeout happened in ReadPipe.
                    return null;
//                    throw new Exception("Timed out while trying to read data.");
                }
                if (data.Length == byteCount) return data;
                if (accumulate == null)
                {
                    accumulate = new byte[byteCount];
                }
                Array.Copy(data, 0, accumulate, read, data.Length);
                read += data.Length;
            }
            return accumulate;
        }

        // basic synchronous read
        public UInt16[] ReadPipeU16(byte pipeId, int count)
        {

            byte[] data = new byte[count*2];

            UInt32 transferSize = 0;
            if (!NativeMethods.WinUsb_ReadPipe(WinusbHandle, pipeId, data, (uint)count*2, ref transferSize, IntPtr.Zero))
            {
                if (Marshal.GetLastWin32Error() == NativeMethods.ERROR_SEM_TIMEOUT)
                {
                    // This was a pipe timeout. Return an empty byte array to indicate this case.
                    return new UInt16[0];
                }
                throw new Exception("ReadPipe failed. " + (new Win32Exception()).ToString());
            }

            UInt16[] newdata = new UInt16[transferSize / 2];
            for (int i = 0; i < (transferSize / 2); i++)
            {
                int v = BitConverter.ToInt16(data, i * 2);
                newdata[i] = (UInt16)v;
            }
            return newdata;

        }

        // basic synchronous read
        public byte[] ReadPipe(byte pipeId, int byteCount)
        {

            byte[] data = new byte[byteCount];

            UInt32 transferSize = 0;
            if (!NativeMethods.WinUsb_ReadPipe(WinusbHandle, pipeId, data, (uint)byteCount, ref transferSize, IntPtr.Zero))
            {
            	return new byte[0];
//                if (Marshal.GetLastWin32Error() == NativeMethods.ERROR_SEM_TIMEOUT)
//                {
//                    // This was a pipe timeout. Return an empty byte array to indicate this case.
//                    
//                }
//                throw new Exception("ReadPipe failed. " + (new Win32Exception()).ToString());
            }

            byte[] newdata = new byte[transferSize];
            Array.Copy(data, newdata, transferSize);
            return newdata;

        }

        // Asynchronous read bits, only for use with buffered reader for now.
        internal void BeginReadPipe(byte pipeId, QueuedBuffer buffer)
        {
            buffer.Overlapped.WaitEvent.Reset();

            if (!NativeMethods.WinUsb_ReadPipe(WinusbHandle, pipeId, buffer.PinnedBuffer, (uint)buffer.BufferSize, IntPtr.Zero, buffer.Overlapped.OverlappedStruct))
            {
                if (Marshal.GetLastWin32Error() != NativeMethods.ERROR_IO_PENDING)
                {
                    throw new Exception("ReadPipe failed. " + (new Win32Exception()).ToString());
                }
            }
        }

        internal byte[] EndReadPipe(QueuedBuffer buf)
        {
            UInt32 transferSize;

            if (!NativeMethods.WinUsb_GetOverlappedResult(WinusbHandle, buf.Overlapped.OverlappedStruct, out transferSize, true))
            {
                if (Marshal.GetLastWin32Error() == NativeMethods.ERROR_SEM_TIMEOUT)
                {
                    // This was a pipe timeout. Return an empty byte array to indicate this case.
                    //System.Diagnostics.Debug.WriteLine("Timed out");
                    return null;
                }
                throw new Exception("ReadPipe's overlapped result failed. " + (new Win32Exception()).ToString());
            }

            byte[] data = new byte[transferSize];
            Marshal.Copy(buf.PinnedBuffer, data, 0, (int)transferSize);
            return data;
        }


        // basic synchronous send.
        public void WritePipe(byte pipeId, byte[] pipeData)
        {

            int remainingbytes = pipeData.Length;
            while (remainingbytes > 0)
            {

                UInt32 transferSize = 0;
                if (!NativeMethods.WinUsb_WritePipe(WinusbHandle, pipeId, pipeData, (uint)pipeData.Length, ref transferSize, IntPtr.Zero))
                {
                    throw new Exception("WritePipe failed. " + (new Win32Exception()).ToString());
                }
                if (transferSize == pipeData.Length) return;

                remainingbytes -= (int)transferSize;

                // Need to retry. Copy the remaining data to a new buffer.
                byte[] data = new byte[remainingbytes];
                Array.Copy(pipeData, transferSize, data, 0, remainingbytes);

                pipeData = data;
            }
        }



        public void ControlTransferOut(byte requestType, byte request, UInt16 value, UInt16 index, byte[] data)
        {
            NativeMethods.WINUSB_SETUP_PACKET setupPacket = new NativeMethods.WINUSB_SETUP_PACKET();
            setupPacket.RequestType = (byte)(requestType | ControlDirectionOut);
            setupPacket.Request = request;
            setupPacket.Value = value;
            setupPacket.Index = index;
            if (data != null)
            {
                setupPacket.Length = (ushort)data.Length;
            }

            UInt32 actualLength = 0;

            if (!NativeMethods.WinUsb_ControlTransfer(WinusbHandle, setupPacket, data, setupPacket.Length, out actualLength, IntPtr.Zero))
            {
            	Dispose();
            	return;
                //throw new Exception("ControlTransfer failed. " + (new Win32Exception()).ToString());
            }
            
            if (data != null && actualLength != data.Length)
            {
                throw new Exception("Not all data transferred");
            }
        }

        public byte[] ControlTransferIn(byte requestType, byte request, UInt16 value, UInt16 index, UInt16 length)
        {
            NativeMethods.WINUSB_SETUP_PACKET setupPacket = new NativeMethods.WINUSB_SETUP_PACKET();
            setupPacket.RequestType = (byte)(requestType | ControlDirectionIn);
            setupPacket.Request = request;
            setupPacket.Value = value;
            setupPacket.Index = index;
            setupPacket.Length = length;

            byte[] output = new byte[length];
            UInt32 actualLength = 0;

            if(!NativeMethods.WinUsb_ControlTransfer(WinusbHandle, setupPacket, output, (uint)output.Length, out actualLength, IntPtr.Zero))
            {
            	Dispose();
            	return null;
//                throw new Exception("ControlTransfer failed. " + (new Win32Exception()).ToString());
            }

            if(actualLength != output.Length)
            {
                byte[] copyTo = new byte[actualLength];
                Array.Copy(output, copyTo, actualLength);
                output = copyTo;
            }
            return output;
        }

        const byte ControlDirectionOut = 0x00;
        const byte ControlDirectionIn = 0x80;

        public const byte ControlTypeStandard = 0x00;
        public const byte ControlTypeClass = 0x20;
        public const byte ControlTypeVendor = 0x40;

        public const byte ControlRecipientDevice = 0;
        public const byte ControlRecipientInterface = 1;
        public const byte ControlRecipientEndpoint = 2;
        public const byte ControlRecipientOther = 3;


    }
    class QueuedBuffer : IDisposable
    {
        public readonly int BufferSize;
        public Overlapped Overlapped;
        public IntPtr PinnedBuffer;
        public QueuedBuffer(int bufferSizeBytes)
        {
            BufferSize = bufferSizeBytes;
            Overlapped = new Overlapped();
            PinnedBuffer = Marshal.AllocHGlobal(BufferSize);
        }

        public void Dispose()
        {
            Overlapped.Dispose();
            Marshal.FreeHGlobal(PinnedBuffer);
            GC.SuppressFinalize(this);
        }

        public void Wait()
        {
            Overlapped.WaitEvent.WaitOne();
        }

        public bool Ready
        {
            get
            {
                return Overlapped.WaitEvent.WaitOne(0);
            }
        }
        
    }
    public interface IPipeByteReader
    {
        /// <summary>
        /// Receive a number of bytes from the incoming data stream.
        /// If there are not enough bytes available, only the available bytes will be returned.
        /// Returns immediately.
        /// </summary>
        /// <param name="count">Number of bytes to request</param>
        /// <returns>Byte data from the USB pipe</returns>
        byte[] ReceiveBytes(int count);

        /// <summary>
        /// Receive a number of bytes from the incoming data stream, but don't remove them from the queue.
        /// If there are not enough bytes available, only the available bytes will be returned.
        /// Returns immediately.
        /// </summary>
        /// <param name="count">Number of bytes to request</param>
        /// <returns>Byte data from the USB pipe</returns>
        byte[] PeekBytes(int count);

        /// <summary>
        /// Receive a specific number of bytes from the incoming data stream.
        /// This call will block until the requested bytes are available, or will eventually throw on timeout.
        /// </summary>
        /// <param name="count">Number of bytes to request</param>
        /// <returns>Byte data from the USB pipe</returns>
        byte[] ReceiveExactBytes(int count);

        /// <summary>
        /// Drop bytes from the incoming data stream without reading them.
        /// If you try to drop more bytes than are available, the buffer will be cleared.
        /// Returns immediately.
        /// </summary>
        /// <param name="count">Number of bytes to drop.</param>
        void SkipBytes(int count);

        /// <summary>
        /// Current number of bytes that are queued and available to read.
        /// </summary>
        int QueuedDataLength { get; }
    }
    public interface IPipePacketReader
    {
        /// <summary>
        /// Number of received packets that can be read.
        /// </summary>
        int QueuedPackets { get; }

        /// <summary>
        /// Retrieve the next packet, but do not remove it from the buffer.
        /// Warning: If you modify the returned array, the modifications will be present in future calls to Peek/Dequeue for this pacekt.
        /// </summary>
        /// <returns>The contents of the next packet in the receive queue</returns>
        byte[] PeekPacket();

        /// <summary>
        /// Retrieve the next packet from the receive queue
        /// </summary>
        /// <returns>The contents of the next packet in the receive queue</returns>
        byte[] DequeuePacket();
    }

    // Background thread to receive data from pipes.
    // Provides two data access mechanisms which are mutually exclusive: Packet level and byte level.
    internal class BufferedPipeThread : IPipeByteReader, IPipePacketReader
    {
        // Logic to enforce interface exclucivity is in WinUSBDevice
        public bool InterfaceBound; // Has the interface been bound?
        public bool PacketInterface; // Are we using the packet reader interface?


        Thread PipeThread;
        WinUSBDevice Device;
        byte DevicePipeId;

        long TotalReceived;

        int QueuedLength;
        Queue<byte[]> ReceivedData;
        int SkipFirstBytes;
        public bool Stopped = false;

        ManualResetEvent ReceiveTick;

        QueuedBuffer[] BufferList;
        Queue<QueuedBuffer> PendingBuffers;

        public BufferedPipeThread(WinUSBDevice dev, byte pipeId, int bufferCount, int multiPacketCount)
        {
            int maxTransferSize = (int)dev.GetPipePolicy(pipeId, WinUsbPipePolicy.MAXIMUM_TRANSFER_SIZE);

            int pipeSize = 512; // Todo: query pipe transfer size for 1:1 mapping to packets.
            int bufferSize = pipeSize * multiPacketCount;
            if (bufferSize > maxTransferSize) { bufferSize = maxTransferSize; }

            PendingBuffers = new Queue<QueuedBuffer>(bufferCount);
            BufferList = new QueuedBuffer[bufferCount];
            for (int i = 0; i < bufferCount;i++)
            {
                BufferList[i] = new QueuedBuffer(bufferSize);
            }

            EventConcurrency = new Semaphore(3, 3);
            Device = dev;
            DevicePipeId = pipeId;
            QueuedLength = 0;
            ReceivedData = new Queue<byte[]>();
            ReceiveTick = new ManualResetEvent(false);
            PipeThread = new Thread(ThreadFunc);
            PipeThread.IsBackground = true;

            //dev.SetPipePolicy(pipeId, WinUsbPipePolicy.PIPE_TRANSFER_TIMEOUT, 1000);

            // Start reading on all the buffers.
            foreach(QueuedBuffer qb in BufferList)
            {
                dev.BeginReadPipe(pipeId, qb);
                PendingBuffers.Enqueue(qb);
            }

            //dev.SetPipePolicy(pipeId, WinUsbPipePolicy.RAW_IO, 1);

            PipeThread.Start();
        }

        public long TotalReceivedBytes { get { return TotalReceived; } }

        //
        // Packet Reader members
        //

        public int QueuedPackets { get { lock (this) { return ReceivedData.Count; } } }

        public byte[] PeekPacket()
        {
            lock (this)
            {
                return ReceivedData.Peek();
            }
        }

        public byte[] DequeuePacket()
        {
            lock (this)
            {
                return ReceivedData.Dequeue();
            }
        }

        //
        // Byte Reader members
        //

        public int QueuedDataLength { get {  return QueuedLength;  } }

        // Only returns as many as it can.
        public byte[] ReceiveBytes(int count)
        {
            int queue = QueuedDataLength;
            if (queue < count) 
                count = queue;

            byte[] output = new byte[count];
            lock (this)
            {
                CopyReceiveBytes(output, 0, count);
            }
            return output;
        }

        // Only returns as many as it can.
        public byte[] PeekBytes(int count)
        {
            int queue = QueuedDataLength;
            if (queue < count)
                count = queue;

            byte[] output = new byte[count];
            lock (this)
            {
                CopyPeekBytes(output, 0, count);
            }
            return output;
        }

        public byte[] ReceiveExactBytes(int count)
        {
            byte[] output = new byte[count];
            if (QueuedDataLength >= count)
            {
                lock (this)
                {
                    CopyReceiveBytes(output, 0, count);
                }
                return output;
            }
            int failedcount = 0;
            int haveBytes = 0;
            while (haveBytes < count)
            {
                ReceiveTick.Reset();
                lock (this)
                {
                    int thisBytes = QueuedLength;

                    if(thisBytes == 0)
                    {
                        failedcount++;
                        if(failedcount > 3)
                        {
                            throw new Exception("Timed out waiting to receive bytes");
                        }
                    }
                    else
                    {
                        failedcount = 0;
                        if (thisBytes + haveBytes > count) thisBytes = count - haveBytes;
                        CopyReceiveBytes(output, haveBytes, thisBytes);
                    }
                    haveBytes += (int)thisBytes;
                }
                if(haveBytes < count)
                {
                    if (Stopped) throw new Exception("Not going to have enough bytes to complete request.");
                    ReceiveTick.WaitOne();
                }
            }
            return output;
        }

        public void SkipBytes(int count)
        {
            lock (this)
            {
                int queue = QueuedLength;
                if (queue < count)
                    throw new ArgumentException("count must be less than the data length");

                int copied = 0;
                while (copied < count)
                {
                    byte[] firstData = ReceivedData.Peek();
                    int available = firstData.Length - SkipFirstBytes;
                    int toCopy = count - copied;
                    if (toCopy > available) toCopy = available;

                    if (toCopy == available)
                    {
                        ReceivedData.Dequeue();
                        SkipFirstBytes = 0;
                    }
                    else
                    {
                        SkipFirstBytes += toCopy;
                    }

                    copied += toCopy;
                    QueuedLength -= toCopy;
                }
            }
        }

        //
        // Internal functionality
        //

        // Must be called under lock with enough bytes in the buffer.
        void CopyReceiveBytes(byte[] target, int start, int count)
        {
            int copied = 0;
            while(copied < count)
            {
                byte[] firstData = ReceivedData.Peek();
                int available = firstData.Length - SkipFirstBytes;
                int toCopy = count - copied;
                if (toCopy > available) toCopy = available;

                Array.Copy(firstData, SkipFirstBytes, target, start, toCopy); 

                if(toCopy == available)
                {
                    ReceivedData.Dequeue();
                    SkipFirstBytes = 0;
                }
                else
                {
                    SkipFirstBytes += toCopy;
                }

                copied += toCopy;
                start += toCopy;
                QueuedLength -= toCopy;
            }
        }

        // Must be called under lock with enough bytes in the buffer.
        void CopyPeekBytes(byte[] target, int start, int count)
        {
            int copied = 0;
            int skipBytes = SkipFirstBytes;

            foreach(byte[] firstData in ReceivedData)
            {
                int available = firstData.Length - skipBytes;
                int toCopy = count - copied;
                if (toCopy > available) toCopy = available;

                Array.Copy(firstData, skipBytes, target, start, toCopy);

                skipBytes = 0;

                copied += toCopy;
                start += toCopy;

                if (copied >= count)
                {
                    break;
                }
            }
        }




        void ThreadFunc(object context)
        {
            Queue<byte[]> receivedData = new Queue<byte[]>(BufferList.Length);

            while(true)
            {
                if (Device.Stopping)
                    break;

                try
                {
                    PendingBuffers.Peek().Wait();
                    // Process a large group of received buffers in a batch, if available.
                    int n = 0;
                    try
                    {
                        while (n < BufferList.Length)
                        {
                            QueuedBuffer buf = PendingBuffers.Peek();
                            if (n == 0 || buf.Ready)
                            {
                                byte[] data = Device.EndReadPipe(buf);
                                PendingBuffers.Dequeue();
                                if (data != null)
                                {   // null is a timeout condition.
                                    receivedData.Enqueue(data);
                                }
                                Device.BeginReadPipe(DevicePipeId, buf);
                                // Todo: If this operation fails during normal operation, the buffer is lost from rotation.
                                // Should never happen during normal operation, but should confirm and mitigate if it's possible.
                                PendingBuffers.Enqueue(buf);

                            }
                            n++;
                        }
                    }
                    finally
                    {
                        // Unless we're exiting, ensure we always indicate the data, even if some operation failed.
                        if(!Device.Stopping && receivedData.Count > 0)
                        {
                            lock (this)
                            {
                                foreach (byte[] data in receivedData)
                                {
                                    ReceivedData.Enqueue(data);
                                    QueuedLength += data.Length;
                                    TotalReceived += data.Length;
                                }
                            }
                            ThreadPool.QueueUserWorkItem(RaiseNewData);
                            receivedData.Clear();
                        }
                    }
                }
                catch(Exception ex)
                {
                    System.Diagnostics.Debug.Print("Should not happen: Exception in background thread. {0}", ex.ToString());
                    Thread.Sleep(15);
                }

                ReceiveTick.Set();

            }
            Stopped = true;
        }

        public event WinUSBDevice.NewDataCallback NewDataEvent;

        Semaphore EventConcurrency;

        void RaiseNewData(object context)
        {
            WinUSBDevice.NewDataCallback cb = NewDataEvent;
            if (cb != null)
            {
                if(EventConcurrency.WaitOne(0)) // Prevent requests from stacking up; Don't issue new events if there are several in flight
                {
                    try
                    {
                        cb();
                    }
                    finally
                    {
                        EventConcurrency.Release();
                    }

                }
            }
        }

    }
    #endregion
    
}
