﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace MiniStm32JoeC_CoreLibrary.Views {
    public partial class UC_PinMap : UserControl {
        MiniStm32BoardJoeC _board;

        public UC_PinMap() {
            InitializeComponent();
            cb_remap_InPullups.SelectedIndex = 0;
            cb_remap_OutType.SelectedIndex = 0;
        }
        public void Init(MiniStm32BoardJoeC miniStm32) {
            _board = miniStm32;
        }
        bool _closing = false;
        public void Close() {
            _closing = true;
            timer_remap.Enabled = false;
        }
        void WaitforBusy(int timeoutInSeconds) {
            DateTime dtTimeout = DateTime.Now.AddSeconds(timeoutInSeconds);
            while (_board.IsBusy) {
                if (DateTime.Now.Ticks > dtTimeout.Ticks) {
                    throw new Exception($"WaitforBusy timeout after '{timeoutInSeconds}'");
                }
                Application.DoEvents();
                System.Threading.Thread.Sleep(10);
            }
        }

        public void SetReferenceOffsets(int offsetForVref, int offsetForTemp) {
            num_RefOffset.Value = offsetForVref;
            num_TempOffset.Value = offsetForTemp;
        }
        public void RefreshBoardInfo(bool withGpioConfig) {
            try {
                WaitforBusy(3);
                _board.IsBusy = true;
                txt_GetId_FW.Text = "";
                txt_GetId_ID.Text = "";
                txt_GetId_Log.Text = "";
                txt_GetId_Vref.Text = "";
                txt_GetId_Temp.Text = "";
                Application.DoEvents();
                BoardInfo boardInfo = _board.GetBoardInfo();

                txt_GetId_FW.Text = $"{boardInfo.FirmwareVersion[0]}.{boardInfo.FirmwareVersion[1]}.{boardInfo.FirmwareVersion[2]}";
                txt_GetId_ID.Text = boardInfo.UniqeID;
                txt_GetId_InitType.Text = boardInfo.InitType.ToString();

                txt_GetId_MainLoopCount.Text = boardInfo.MainLoopCount.ToString();
                txt_GetId_ActiveTime.Text = _board.CalculateTimeStringFromTotalSeconds(boardInfo.ActiveSeconds);

                if (withGpioConfig) {
                    PortInfo info = _board.GetPortInfo("A");
                    RefreshBoardConfig(info);
                    info = _board.GetPortInfo("B");
                    RefreshBoardConfig(info);
                    info = _board.GetPortInfo("C");
                    RefreshBoardConfig(info);
                }

                ushort[] raws = _board.GetTemperatureAndReferenceVoltageRaw();
                AddToLog($"Get raw: ref'{raws[0]}' temp'{raws[1]}'");
                raws[0] = (ushort)(raws[0] - num_RefOffset.Value);
                raws[1] = (ushort)(raws[1] - num_TempOffset.Value);
                AddToLog($"\r\nwith offset: ref'{raws[0]}' temp'{raws[1]}'");
                string log = "";
                double[] refAndTemp = _board.CalculateTemperatureAndReferenceVoltage(raws, ref log);
                txt_GetId_Vref.Text = Math.Round(refAndTemp[0], 2).ToString() + " V";
                txt_GetId_Temp.Text = Math.Round(refAndTemp[1], 1).ToString() + " °C";
                AddToLog("\r\n" + log);
                _board.IsBusy = false;
            }
            catch (Exception ex) {
                _board.IsBusy = false;
                if (_closing) { return; }
                txt_GetId_Log.Text += "Ex: " + ex.Message;
                throw ex;
            }
        }

        //boardconfig
        public void RefreshBoardConfig(PortInfo port) {
            switch (port.Port) {
                case "A": Dgv_Gpio_A.Rows.Clear(); break;
                case "B": Dgv_Gpio_B.Rows.Clear(); break;
                case "C": Dgv_Gpio_C.Rows.Clear(); break;
                default: return;
            }
            GpioReadConfig_to_Dgv(port);
        }
        public void AddToLog(string info) {
            txt_GetId_Log.Text += info;
        }
        void GpioReadConfig_to_Dgv(PortInfo portInfo) {
            Color PortColor = Color.Fuchsia;
            DataGridView dgv = Dgv_Gpio_A;
            switch (portInfo.Port) {
                case "A": PortColor = Color.Gold; dgv = Dgv_Gpio_A; break;
                case "B": PortColor = Color.RoyalBlue; dgv = Dgv_Gpio_B; break;
                case "C": PortColor = Color.Green; dgv = Dgv_Gpio_C; break;
            }
            for (int i = 0; i < portInfo.Pins.Count; i++) {
                PinInfo pin = portInfo.Pins[i];
                string info = $"{pin.EnumPinMode}";
                switch (pin.EnumPinMode) {
                    case PinMode.OP_Out_PushPull:
                    case PinMode.OD_Out_OpenDrain:
                    case PinMode.AP_Alt_PushPull:
                    case PinMode.AD_Alt_OpenDrain:
                        switch (pin.EnumPinSpeed) {
                            case PinHwSpeed.Mhz_2: info = $"{pin.EnumPinMode}_2Mhz"; break;
                            case PinHwSpeed.Mhz_10: info = $"{pin.EnumPinMode}_10Mhz"; break;
                            case PinHwSpeed.Mhz_50: info = $"{pin.EnumPinMode}_50Mhz"; break;
                        }
                        break;
                }

                dgv.Rows.Add(pin.Label, info,"","");
                dgv.Rows[dgv.RowCount - 1].Cells[0].Style.BackColor = PortColor;
                dgv.Rows[dgv.RowCount - 1].Cells[2].Style.BackColor = (pin.EnumPinOdrState == PinLogicalState.High) ? Color.Red : Color.Gainsboro;
                dgv.Rows[dgv.RowCount - 1].Cells[3].Style.BackColor = (pin.EnumPinIdrState == PinLogicalState.High) ? Color.LimeGreen : Color.Gainsboro;
            }
            dgv.ClearSelection();
        }
        
        //remap
        void timer_remap_Tick(object sender, EventArgs e) {
            if (btn_remap_ReadSel.BackColor == Color.Gold) { return; }
            btn_remap_ReadSel_Click(null, null);
        }
        void btn_remap_ReadSel_Click(object sender, EventArgs e) {
            btn_remap_ReadSel.BackColor = Color.Gold; btn_remap_ReadSel.Refresh();
            try {
                WaitforBusy(3);
                _board.IsBusy = true;
                StringBuilder sbrem = new StringBuilder();
                int pinnr = int.Parse(txt_remap_Pin.Text.Remove(0, 1));

                if (rad_remap_Din.Checked) {
                    bool value = _board.GetPinInputState($"{txt_remap_Pin.Text[0]}{pinnr.ToString("X")}");
                    if (value) {
                        txt_remap_result.BackColor = Color.Salmon;
                        txt_remap_result.Text = "High";
                    }
                    else {
                        txt_remap_result.BackColor = Color.PaleGreen;
                        txt_remap_result.Text = "Low";
                    }
                }
                if (rad_remap_Ain.Checked) {
                    ushort value = _board.GetPinAdc($"{txt_remap_Pin.Text[0]}{pinnr.ToString("X")}");
                    txt_remap_result.BackColor = Color.White;
                    txt_remap_result.Text = value.ToString();
                }
                if (rad_remap_Dout.Checked) {
                    chk_remap_DoutState.Checked = !chk_remap_DoutState.Checked;
                    chk_remap_DoutState.Refresh();
                    if (chk_remap_DoutState.Checked) {
                        _board.SetPinLine(new string[] { $"{txt_remap_Pin.Text[0]}{pinnr.ToString("X")}" }, PinSetState.PinState_PP_High);
                    }
                    else {
                        _board.SetPinLine(new string[] { $"{txt_remap_Pin.Text[0]}{pinnr.ToString("X")}" }, PinSetState.PinState_PP_Low);
                    }
                }
                _board.IsBusy = false;
            }
            catch (Exception ex) {
                _board.IsBusy = false;
                if (_closing) { return; }
                btn_remap_ReadSel.BackColor = Color.Red; btn_remap_ReadSel.Refresh();
                MessageBox.Show(ex.Message, "Read remap pin");
            }
            btn_remap_ReadSel.BackColor = Color.Gainsboro; btn_remap_ReadSel.Refresh();
        }
        void btn_remap_Pin_Click(object sender, EventArgs e) {
            chk_remap_timer.Checked = false;
            btn_remap_Pin.BackColor = Color.Gold; btn_remap_Pin.Refresh();
            try {
                WaitforBusy(3);
                _board.IsBusy = true;
                PinSetState setState = PinSetState.Undefined;

                if (rad_remap_Din.Checked) {
                    switch (cb_remap_InPullups.SelectedIndex) {
                        case 1: setState = PinSetState.PinState_In_PDown; break;
                        case 2: setState = PinSetState.PinState_In_PUp; break;
                        default: setState = PinSetState.PinState_In_Float; break;
                    }
                }
                if (rad_remap_Ain.Checked) {
                    setState = PinSetState.PinState_In_Analog;
                }
                if (rad_remap_Dout.Checked) {
                    switch (cb_remap_OutType.SelectedIndex) {
                        case 0: //PushPull
                            setState = (chk_remap_DoutState.Checked) ? PinSetState.PinState_PP_High : PinSetState.PinState_PP_Low;
                            break;
                        case 1: //OpenDrain
                            setState = (chk_remap_DoutState.Checked) ? PinSetState.PinState_OD_High : PinSetState.PinState_OD_Low;
                            break;
                        case 2: //Alt_PP
                            setState = (chk_remap_DoutState.Checked) ? PinSetState.PinState_AP_High : PinSetState.PinState_AP_Low;
                            break;
                        case 3: //Alt_OD
                            setState = (chk_remap_DoutState.Checked) ? PinSetState.PinState_AD_High : PinSetState.PinState_AD_Low;
                            break;
                    }
                }
                //execute
                if (setState != PinSetState.Undefined) {
                    string[] pins = GetRemapPins();
                    if (pins == null) {
                        throw new Exception("GetRemapPins()->null");
                    }
                    _board.SetPinLine(pins, setState);
                }
                _board.IsBusy = false;
            }
            catch (Exception ex) {
                _board.IsBusy = false;
                if (_closing) { return; }
                btn_remap_Pin.BackColor = Color.Red; btn_remap_Pin.Refresh();
                MessageBox.Show(ex.Message, "remap pin");
                return;
            }
            btn_remap_Pin.BackColor = Color.Gainsboro; btn_remap_Pin.Refresh();
        }
        string[] GetRemapPins() {
            List<string> pins = new List<string>();
            string[] slits = txt_remap_Pin.Text.Split(',');
            foreach (var item in slits) {
                int pinnr = 0;
                int.TryParse(item.Remove(0, 1), out pinnr);
                if (pinnr < 0 || 15 < pinnr) {
                    continue; // invalid
                }
                pins.Add($"{item[0]}{pinnr.ToString("X")}");
            }
            if (pins.Count == 0) {
                return null;
            }
            return pins.ToArray();
        }
        void chk_remap_timer_CheckedChanged(object sender, EventArgs e) {
            timer_remap.Enabled = chk_remap_timer.Checked;
            btn_remap_ReadSel.Enabled = !chk_remap_timer.Checked;
        }
        void num_remap_Timer_ValueChanged(object sender, EventArgs e) {
            timer_remap.Interval = (int)num_remap_Timer.Value;
        }
    }
}
