﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading;

namespace MiniStm32JoeC_CoreLibrary {
    public class ScriptCommandInterpreter {
        MiniStm32BoardJoeC _board;
        #region base
        public ScriptCommandInterpreter(MiniStm32BoardJoeC board) {
            _board = board;
        }
        public event Action<string, Color> Event_AddLog;
        public void OnEvent_AddLog(string info, Color color) {
            if (Event_AddLog != null) { Event_AddLog(info, color); }
        }
        public int GetIndexForCmd(string cmd) {
            int cnt = ScriptCmds.Length / NrOfColumns; //nr of columns
            for (int r = 0; r < cnt; r++) {
                if (ScriptCmds[r, 0] == cmd) {
                    return r;
                }
            }
            return -1;
        }
        public string ShowHelp() {
            StringBuilder sb = new StringBuilder();
            int cnt = ScriptCmds.Length / NrOfColumns; //nr of columns
            sb.AppendLine("Nr\tCmd\tParameter\tNotes");
            for (int r = 0; r < cnt; r++) {
                sb.AppendLine($"{r}\t{ScriptCmds[r, 0]}\t{ScriptCmds[r, 1]}\t{ScriptCmds[r, 2]}");
            }
            sb.AppendLine("Syntax: <Cmd>:<Para1>,<Para2>... #comment (ignored)");
            return sb.ToString();
        }
        #endregion

        public void Execute(string cmd) {
            string[] splits = cmd.Split(':');
            if (splits.Length != 2) {
                throw new Exception("Execute() -> Syntax error, need one ':'.");
            }
            int index = GetIndexForCmd(splits[0]);
            if (index < 0) {
                //we missing entry in: string[,] ScriptCmds
                throw new Exception($"Execute() -> Command '{splits[0]}' not known.");
            }
            switch (splits[0]) {
                case "DOUT": ScriptCmd_DOUT(splits[1]); break;
                case "LED": ScriptCmd_LED(splits[1]); break;
                case "SPITARGET": ScriptCmd_SPITARGET(splits[1]); break;
                case "SPI8": ScriptCmd_SPI8(splits[1]); break;
                case "SPI16": ScriptCmd_SPI16(splits[1]); break;
                case "SPI32": ScriptCmd_SPI32(splits[1]); break;
                case "I2C": ScriptCmd_I2C(splits[1]); break;
                //handled before
                case "SLEEP":
                case "MSGBOX":
                case "SPI<Bits>":
                    break;
                default:
                    throw new Exception($"Execute() -> Command '{splits[0]}' not defined.");
            }
        }
        int NrOfColumns = 3;
        public string[,] ScriptCmds = new string[,] {
            //<CMD>,<parameter syntax>,<descr>
            //first commands are handled in user control, not here
            { "SLEEP","<time in ms>", "wait some time"},
            { "MSGBOX","<string>", "Show Message box and ask for continue execution."},
            //all below are handled here(except first, its only for info)
            { "SPITARGET","<target>", "set target for SPI transmissions"},
            { "SPI<Bits>","<D=Dec,H=Hex,none=Hex>,", "first 2 chars may define type of data<D=Dec,H=Hex,none=Hex>."},
            { "SPI8","D,<byte1>,<byte2>", "perform spi transmission, each block in decimal byte."},
            { "SPI16","<ushort1>,<ushort2>", "perform spi transmission, each block as hex value."},
            { "SPI32","H,<uint32_1>,<uint32_2>", "perform spi transmission, each block as hex value."},
            { "LED", "<red>,<green>,<blue>,<nr of leds>" , "write single color to digital RGB led"},
            { "DOUT", "<OP|OD>,B12,<0|L|F or 1|H|T>" , "set digital output, interprete first char as state(L or Low or F or False -> 0)"},
            { "I2C", "<R>,<Add>,<Cnt>" , "R=Read"},
            { "I2C", "<W>,<Add>,<byte0>,<byte1>..." , "W=Write"},
            { "I2C", "<S>,<Cnt>,<Mode>", "S=Scan... loop any address and list ACK (Chip response)"},
            { "I2C", "<T>,<Add>,<TxCnt>,<RxCnt>,<byte0>,<byte1>..." , "T=WriteRead"},
        };

        //Script commands ###############################
        #region Commands
        void ScriptCmd_LED(string parameter) {
            string[] splits = parameter.Split(',');
            byte[] para = new byte[4];
            if (splits.Length < para.Length) {
                throw new Exception($"LED()->Need '{para.Length}' parameters, but have only '{splits.Length}'.");
            }
            for (int i = 0; i < para.Length; i++) {
                if (!byte.TryParse(splits[i], out para[i])) {
                    throw new Exception($"LED()->cannot parse '{splits[i]}' to an byte.");
                }
            }
            _board.SetRgbLed_SingleColor(para[0], para[1], para[2], para[3]);
        }
        void ScriptCmd_SPITARGET(string parameter) {
            _board.SpiTarget = byte.Parse(parameter);
        }
        void ScriptCmd_SPI8(string parameter) {
            int dtype = 0;
            if (parameter.StartsWith("D,")) { dtype = 1; }
            if (parameter.StartsWith("H,")) { dtype = 2; }
            if (dtype == 0) {
                dtype = 2; //default hex
            } else {
                parameter = parameter.Remove(0, 2);
            }
            string dtypestr = (dtype == 1) ? "DEC" : "HEX";

            //Build and perform Transmission
            string[] splits = parameter.TrimEnd().Split(',');
            byte[] tx = new byte[splits.Length];

            StringBuilder sbDataIn = new StringBuilder();
            if (dtype == 1) {
                for (int i = 0; i < tx.Length; i++) {
                    if (!byte.TryParse(splits[i], out tx[i])) {
                        throw new Exception($"SPI8()->Can't parse '{splits[i]}' to a byte");
                    }
                    sbDataIn.Append($"{tx[i]},");
                }
            }
            if (dtype == 2) {
                for (int i = 0; i < tx.Length; i++) {
                    if (!byte.TryParse(splits[i], System.Globalization.NumberStyles.HexNumber, null, out tx[i])) {
                        throw new Exception($"SPI8()->Can't parse '{splits[i]}' to a byte");
                    }
                    sbDataIn.Append($"{tx[i].ToString("X02")},");
                }
            }
            byte[] rx = _board.TransmissionSpi8(_board.SpiTarget, tx.Length, tx);
            if (rx == null) { OnEvent_AddLog($"Spi8({dtypestr}):{sbDataIn.ToString()} -> null (transmission error)", Color.Red); return; }

            StringBuilder sbDataOut = new StringBuilder();
            if (dtype == 1) {
                for (int i = 0; i < rx.Length; i++) {
                    sbDataOut.Append($"{rx[i]},");
                }
            }
            if (dtype == 2) {
                for (int i = 0; i < rx.Length; i++) {
                    sbDataOut.Append($"{rx[i].ToString("X02")},");
                }
            }
            OnEvent_AddLog($"Spi8({dtypestr}): {sbDataIn.ToString()} -> {sbDataOut.ToString()}", Color.Black);
        }
        void ScriptCmd_SPI16(string parameter) {
            int dtype = 0;
            if (parameter.StartsWith("D,")) { dtype = 1; }
            if (parameter.StartsWith("H,")) { dtype = 2; }
            if (dtype == 0) {
                dtype = 2; //default hex
            } else {
                parameter = parameter.Remove(0, 2);
            }
            string dtypestr = (dtype == 1) ? "DEC" : "HEX";

            //Build and perform Transmission
            string[] splits = parameter.TrimEnd().Split(',');
            ushort[] tx = new ushort[splits.Length];

            StringBuilder sbDataIn = new StringBuilder();
            if (dtype == 1) {
                for (int i = 0; i < tx.Length; i++) {
                    if (!ushort.TryParse(splits[i], out tx[i])) {
                        throw new Exception($"SPI16()->Can't parse '{splits[i]}' to a ushort");
                    }
                    sbDataIn.Append($"{tx[i]},");
                }
            }
            if (dtype == 2) {
                for (int i = 0; i < tx.Length; i++) {
                    if (!ushort.TryParse(splits[i], System.Globalization.NumberStyles.HexNumber, null, out tx[i])) {
                        throw new Exception($"SPI16()->Can't parse '{splits[i]}' to a ushort");
                    }
                    sbDataIn.Append($"{tx[i].ToString("X04")},");
                }
            }
            ushort[] rx = _board.TransmissionSpi16(_board.SpiTarget, tx.Length, tx);
            if (rx == null) { OnEvent_AddLog($"Spi16({dtypestr}): {sbDataIn.ToString()} -> null (transmission error)", Color.Red); return; }

            StringBuilder sbDataOut = new StringBuilder();
            if (dtype == 1) {
                for (int i = 0; i < rx.Length; i++) {
                    sbDataOut.Append($"{rx[i]},");
                }
            }
            if (dtype == 2) {
                for (int i = 0; i < rx.Length; i++) {
                    sbDataOut.Append($"{rx[i].ToString("X04")},");
                }
            }
            OnEvent_AddLog($"Spi16({dtypestr}): {sbDataIn.ToString()} -> {sbDataOut.ToString()}", Color.Black);
        }
        void ScriptCmd_SPI32(string parameter) {
            int dtype = 0;
            if (parameter.StartsWith("D,")) { dtype = 1; }
            if (parameter.StartsWith("H,")) { dtype = 2; }
            if (dtype == 0) {
                dtype = 2; //default hex
            } else {
                parameter = parameter.Remove(0, 2);
            }
            string dtypestr = (dtype == 1) ? "DEC" : "HEX";

            //Build and perform Transmission
            string[] splits = parameter.TrimEnd().Split(',');
            UInt32[] tx = new UInt32[splits.Length];

            StringBuilder sbDataIn = new StringBuilder();
            if (dtype == 1) {
                for (int i = 0; i < tx.Length; i++) {
                    if (!UInt32.TryParse(splits[i], out tx[i])) {
                        throw new Exception($"SPI32()->Can't parse '{splits[i]}' to a UInt32");
                    }
                    sbDataIn.Append($"{tx[i]},");
                }
            }
            if (dtype == 2) {
                for (int i = 0; i < tx.Length; i++) {
                    if (!UInt32.TryParse(splits[i], System.Globalization.NumberStyles.HexNumber, null, out tx[i])) {
                        throw new Exception($"SPI32()->Can't parse '{splits[i]}' to a UInt32");
                    }
                    sbDataIn.Append($"{tx[i].ToString("X08")},");
                }
            }
            UInt32[] rx = _board.TransmissionSpi32(_board.SpiTarget, tx.Length, tx);
            if (rx == null) { OnEvent_AddLog($"Spi32({dtypestr}): {sbDataIn.ToString()} -> null (transmission error)", Color.Red); return; }

            StringBuilder sbDataOut = new StringBuilder();
            if (dtype == 1) {
                for (int i = 0; i < rx.Length; i++) {
                    sbDataOut.Append($"{rx[i]},");
                }
            }
            if (dtype == 2) {
                for (int i = 0; i < rx.Length; i++) {
                    sbDataOut.Append($"{rx[i].ToString("X08")},");
                }
            }
            OnEvent_AddLog($"Spi32({dtypestr}): {sbDataIn.ToString()} -> {sbDataOut.ToString()}", Color.Black);
        }
        void ScriptCmd_DOUT(string parameter) {
            string[] splits = parameter.Split(',');
            if (splits.Length < 3) {
                throw new Exception($"DOUT()->Need '3' parameters, but have only '{splits.Length}'.");
            }
            PinInfo pi = new PinInfo(splits[1]);
            pi.EnumPinMode = PinMode.OP_Out_PushPull;
            switch (splits[0]) {
                case "OP": pi.EnumPinMode = PinMode.OP_Out_PushPull; break;
                case "OD": pi.EnumPinMode = PinMode.OD_Out_OpenDrain; break;
            }
            switch (splits[2][0]) {
                case '0': case 'L': case 'F':
                    _board.SetDigitalPin(pi,false);
                    break;
                case '1': case 'H': case 'T':
                    _board.SetDigitalPin(pi, true);
                    break;
                default:
                    throw new Exception($"DOUT->state not defined '{splits[2]}'");
            }
        }
        void ScriptCmd_I2C(string parameter) {
            int dtype = 0;
            if (parameter.StartsWith("R,")) { dtype = 1; }
            if (parameter.StartsWith("W,")) { dtype = 2; }
            if (parameter.StartsWith("S,")) { dtype = 3; }
            if (parameter.StartsWith("T,")) { dtype = 4; }
            parameter = parameter.Remove(0, 2);

            //Build and perform Transmission
            string[] splits = parameter.TrimEnd().Split(',');
            byte[] tx = new byte[splits.Length];
            StringBuilder sbDataIn = new StringBuilder();
            for (int i = 0; i < tx.Length; i++) {
                if (splits[i].Length == 0) {
                    continue;
                }
                if (!byte.TryParse(splits[i], System.Globalization.NumberStyles.HexNumber, CultureInfo.InvariantCulture, out tx[i])) {
                    throw new Exception($"I2C()->Can't parse '{splits[i]}' to a byte");
                }
                sbDataIn.Append($"{tx[i]},");
            }
            //byte[] rx = null;
            //if (dtype == 1) {
            //    rx = _board.I2CReadDirect(tx[0],tx[1]);
            //}
            //if (dtype == 2) {
            //    byte[] Wtx = new byte[tx[1]];
            //    Array.Copy(tx, 2, Wtx, 0, Wtx.Length);
            //    rx = _board.I2CWriteDirect(tx[0], Wtx);
            //}
            //if (dtype == 3) {
            //    rx = _board.I2CScanSlavesDirect(tx[0], tx[1]);
            //}
            //if (dtype == 4) {
            //    byte[] Wtx = new byte[tx[1]];
            //    Array.Copy(tx, 3, Wtx, 0, Wtx.Length);
            //    rx = _board.I2CWriteReadDirect(tx[0], tx[2], Wtx);
            //}
            I2cResponseInfo ri = new I2cResponseInfo();
            if (dtype == 1) {
                ri = _board.I2CRead(tx[0], tx[1]);
            }
            if (dtype == 2) {
                byte[] Wtx = new byte[tx.Length-2];
                Array.Copy(tx, 1, Wtx, 0, Wtx.Length);
                ri = _board.I2CWrite(tx[0], Wtx);
            }
            if (dtype == 3) {
                ri = _board.I2CScanSlaves(tx[0], tx[1]);
            }
            if (dtype == 4) {
                byte[] Wtx = new byte[tx[1]];
                Array.Copy(tx, 3, Wtx, 0, Wtx.Length);
                ri = _board.I2CWriteRead(tx[0], tx[2], Wtx);
            }

            if (ri.I2cStateByte == 0) { OnEvent_AddLog($"I2C({dtype}):{sbDataIn.ToString()} -> null (transmission error)", Color.Red); return; }
            //StringBuilder sbDataOut = new StringBuilder();
            //for (int i = 0; i < rx.Length; i++) {
            //    sbDataOut.Append($"{rx[i].ToString("X02")},");
            //}
            OnEvent_AddLog($"I2C({dtype}): {sbDataIn.ToString()} -> {ri.GetResponse(true)}", Color.Black);
        }
        #endregion
    }

}
