﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace FlashLibStm32Uart {
    public partial class UC_FlashLoaderStm32 : UserControl {
        public FlashStm32 Flasher;
        public string LastResults = "";
        List<Button> SequenceButtons = new List<Button>();
        #region FormStuff
        public UC_FlashLoaderStm32() {
            InitializeComponent();
            SequenceButtons.Add(btn_S1_Init);
            SequenceButtons.Add(btn_S2_ReadIds);
            SequenceButtons.Add(btn_S3_ReadFirmwareFile);
            SequenceButtons.Add(btn_S4_EraseChip);
            SequenceButtons.Add(btn_S5_FlashFirmware);
            SequenceButtons.Add(btn_S6_DownloadFirmware);
            SequenceButtons.Add(btn_S7_ExecuteFirmware);
        }
        public void Init(SerialPort sP, bool portStayOpen, bool selectLastChangedBinary) {
            Flasher = new FlashStm32(sP);
            Flasher.PortStayOpen = portStayOpen;
            Flasher.EventReportProgress += Flasher_EventReportProgress;
            Flasher.EventReportMessage += Flasher_EventReportMessage;
            cb_PageSize.SelectedIndex = 1;

            Flasher.IsDebugMessage = chk_DebugMessages.Checked;
            Flasher.SingleByteMode = chk_SingleByteMode.Checked;

            openFileDialog1.InitialDirectory = Application.StartupPath;
            if (selectLastChangedBinary) {
                SelectLastChangedBinary();
            }
        }

        void Flasher_EventReportMessage(string obj) {
            if (InvokeRequired) {
                this.BeginInvoke(new Action<string>(AddLog), obj);
                return;
            }
            AddLog(obj);
        }
        void Flasher_EventReportProgress(float obj) {
            if (InvokeRequired) {
                this.BeginInvoke(new Action<float>(UpdateProgress), obj);
                return;
            }
            UpdateProgress(obj);
        }
        void UpdateProgress(float percent) {
            if (percent <= 100f) {
                BootloaderActionsProgBar.Value = (int)percent;
            }
            BootloaderActionsLblPercent.Text = $"{Math.Round(percent,1)} %";
            BootloaderActionsProgBar.Refresh();
            BootloaderActionsLblPercent.Refresh();
        }
        void SetBtnBackgroundColor(object sender, Color color) {
            Button button = sender as Button;
            button.BackColor = color;
            button.Refresh();
        }
        void AddLog(string message) {
            txtLog.Text += $"{message}\r\n";
            txtLog.Refresh();
        }
        bool RunSingleStep(Button button) {
            button.PerformClick();
            while (button.BackColor == Color.Gold) {
                if (Flasher.IsErrorExisting) {
                    button.BackColor = Color.Fuchsia;
                    System.Threading.Thread.Sleep(100);
                    return false;
                }
                Application.DoEvents();
            }
            System.Threading.Thread.Sleep(100);
            if (button.BackColor == Color.PaleGreen) {
                return true;
            }
            return false;
        }
        void chk_DebugMessages_CheckedChanged(object sender, EventArgs e) {
            Flasher.IsDebugMessage = chk_DebugMessages.Checked;
        }
        void chk_SingleByteMode_CheckedChanged(object sender, EventArgs e) {
            Flasher.SingleByteMode = chk_SingleByteMode.Checked;
        }
        void btn_ClearLog_Click(object sender, EventArgs e) {
            txtLog.Text = "";
        }
        void txt_Address_KeyDown(object sender, KeyEventArgs e) {
            if (e.KeyCode != Keys.Enter) {
                return;
            }
            uint addr;
            try {
                addr = Convert.ToUInt32(txt_Address.Text, 16);
            }
            catch (OverflowException) {
                MessageBox.Show("Address too large!");
                addr = Flasher.BaseAddress;
            }
            catch (Exception) {
                MessageBox.Show("Incorrect hex value!");
                addr = Flasher.BaseAddress;
            }

            /* store realigned address */
            Flasher.Address = addr & 0xffffff00;
            /* start page - end page */
            Flasher.PageSize = (Flasher.Address - Flasher.BaseAddress) / 256;

            txt_Address.Text = string.Format("0x{0:X8}", Flasher.Address);
        }
        void btn_Cancel_Click(object sender, EventArgs e) {
            Flasher.Cancel = true;
            AddLog("Cancel...");
        }
        void cb_PageSize_SelectedIndexChanged(object sender, EventArgs e) {
            Flasher.PageSize = uint.Parse(cb_PageSize.SelectedItem as string);
        }
        void UpdateStatus(string text) {
            AddLog("Status: "+text);
            BootloaderActionsTxtStatus.Text = text;
            BootloaderActionsTxtStatus.Refresh();
            //if (ding) {
            //    /* ^^ ding! */
            //    SystemSounds.Exclamation.Play();
            //}
        }
        public void SetFirmwareFile(string filePath) {
            txt_FirmwarePath.Text = filePath;
            Flasher.StepReadBinary(filePath);
            txt_BinSize.Text = Flasher.FileBuffer.Length.ToString();
            AddLog("ReadBinary: " + Flasher.FileBuffer.Length);
        }
        #endregion
        #region Functions
        public void SelectLastChangedBinary() {
            try {
                string[] files = Directory.GetFiles(Application.StartupPath, "*.bin");
                DateTime LastSelected = DateTime.Parse("1.1.1900");
                foreach (var item in files) {
                    DateTime lastModified = File.GetLastWriteTime(item);
                    if (LastSelected.Ticks < lastModified.Ticks) {
                        txt_FirmwarePath.Text = item;
                        LastSelected = lastModified;
                    }
                }
            }
            catch (Exception ex) {
                AddLog("Ex: " + ex.Message);
            }
        }
        public bool RunSingleSequence() { 
            txtLog.Text = "";
            SetBtnBackgroundColor(btn_RunSingleSequence, Color.Gold);
            foreach (var item in SequenceButtons) {
                SetBtnBackgroundColor(item, Color.Gainsboro);
            }

            //Application.DoEvents();
            for (int i = 0; i < SequenceButtons.Count; i++) {
                switch (i) {
                    case 0: if (!chk_S1.Checked) { continue; } break;
                    case 1: if (!chk_S2.Checked) { continue; } break;
                    case 2: if (!chk_S3.Checked) { continue; } break;
                    case 3: if (!chk_S4.Checked) { continue; } break;
                    case 4: if (!chk_S5.Checked) { continue; } break;
                    case 5: if (!chk_S6.Checked) { continue; } break;
                    case 6: if (!chk_S7.Checked) { continue; } break;
                }
                bool isPass = false;
                try {
                    isPass = RunSingleStep(SequenceButtons[i]);
                }
                catch (Exception ex) {
                    AddLog($"RunSingleStep->{ex.Message}");
                }
                if (!isPass) {
                    AddLog("Cancel sequence here...");
                    if (Flasher.IsErrorExisting) {
                        AddLog(Flasher.GetErrors());
                    }
                    SetBtnBackgroundColor(btn_RunSingleSequence, Color.Salmon);
                    return false;
                }
            }
            SetBtnBackgroundColor(btn_RunSingleSequence, Color.PaleGreen);
            return true;
        }
        void btn_RunSingleSequence_Click(object sender, EventArgs e) {
            RunSingleSequence();
        }

        void btn_S1_Init_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            UpdateStatus("Init Bootloader...");
            bool isinit = Flasher.StepInit();
            if (isinit) {
                LastResults = "Init pass";
                SetBtnBackgroundColor(sender, Color.PaleGreen);
            }
            else {
                LastResults = "Init fail";
                SetBtnBackgroundColor(sender, Color.Salmon);
                AddLog(Flasher.GetErrors());
            }
        }
        void btn_S2_ReadIds_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            UpdateStatus("Read IDs...");
            try {
                Flasher.StepGetVersion();
                AddLog($"GetVersion: {Flasher.Version}");
                Flasher.StepGetProductID();
                AddLog($"ProductID: {Flasher.ProductID} (0x{Flasher.ProductID.ToString("X04")})");
                SetBtnBackgroundColor(sender, Color.PaleGreen);
            }
            catch (Exception ex) {
                LastResults = $"Read IDs->Error:{ex.Message}";
                AddLog("ERROR: " + ex.Message);
                SetBtnBackgroundColor(sender, Color.Salmon);
            }
        }
        void btn_S3_ReadFirmwareFile_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            UpdateStatus("Read binary file...");
            try {
                SetFirmwareFile(txt_FirmwarePath.Text);
                LastResults = $"Read FirmwareFile pass";
                SetBtnBackgroundColor(sender, Color.PaleGreen);
            }
            catch (Exception ex) {
                LastResults = $"Read FirmwareFile->Error:{ex.Message}";
                AddLog("ERROR: " + ex.Message);
                SetBtnBackgroundColor(sender, Color.Salmon);
            }
        }
        void btn_S4_EraseChip_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            try {
                Flasher.Address = Convert.ToUInt32(txt_Address.Text, 16);
                if (chk_GlobalErase.Checked) {
                    UpdateStatus("Erase global chip...");
                    Flasher.StepErase(true);
                }
                else {
                    UpdateStatus("Erase used pages...");
                    if (Flasher.FileBuffer == null) {
                        throw new Exception("no firmware file loaded. FileBuffer is null.");
                    }
                    Flasher.StepErase(false);
                }

                SetBtnBackgroundColor(sender, Color.PaleGreen);
            }
            catch (Exception ex) {
                LastResults = $"EraseChip->Error:{ex.Message}";
                AddLog("ERROR: " + ex.Message);
                SetBtnBackgroundColor(sender, Color.Salmon);
            }
        }
        void btn_S5_FlashFirmware_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            try {
                Flasher.Address = Convert.ToUInt32(txt_Address.Text, 16);
                UpdateStatus($"Programming: 0x{Flasher.Address.ToString("X08")}...");
                Flasher.StepWriteMemory();
                SetBtnBackgroundColor(sender, Color.PaleGreen);
            }
            catch (Exception ex) {
                LastResults = $"FlashFirmware->Error:{ex.Message}";
                AddLog("ERROR: " + ex.Message);
                SetBtnBackgroundColor(sender, Color.Salmon);
            }
        }
        void btn_S6_DownloadFirmware_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            try {
                int nrOfBytes = int.Parse(txt_BinSize.Text);
                UpdateStatus($"Download: {nrOfBytes}...");
                Flasher.ReadMemory(nrOfBytes);

                string newFilename = Path.GetDirectoryName(txt_FirmwarePath.Text) + "\\" + txt_DownloadedBinary.Text;
                Flasher.StepWriteBinary(newFilename);

                if (Flasher.VerifyBuffers()) {
                    SetBtnBackgroundColor(sender, Color.PaleGreen);
                }
                else {
                    SetBtnBackgroundColor(sender, Color.Fuchsia);
                }
                
            }
            catch (Exception ex) {
                LastResults = $"DownloadFirmware->Error:{ex.Message}";
                AddLog("ERROR: " + ex.Message);
                SetBtnBackgroundColor(sender, Color.Salmon);
            }
        }
        void btn_S7_ExecuteFirmware_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            try {
                Flasher.Address = Convert.ToUInt32(txt_Address.Text, 16);
                UpdateStatus($"Execute: 0x{Flasher.Address.ToString("X08")}...");
                Flasher.StepExecute(Flasher.Address);
                SetBtnBackgroundColor(sender, Color.PaleGreen);
            }
            catch (Exception ex) {
                LastResults = $"ExecuteFirmware->Error:{ex.Message}";
                AddLog("ERROR: " + ex.Message);
                SetBtnBackgroundColor(sender, Color.Salmon);
            }
        }
        void btn_ExecuteFirmware_Click(object sender, EventArgs e) {
            SetBtnBackgroundColor(sender, Color.Gold);
            try {
                Flasher.Address = Convert.ToUInt32(txt_Address.Text, 16);
                if (!Flasher.StepInit()) {
                    throw new Exception("StepInit-> fail");
                }
                Flasher.StepGetVersion();
                Flasher.StepGetProductID();
                AddLog($"Connected: Ver: {Flasher.Version}, PID: {Flasher.ProductID}");

                UpdateStatus($"Execute: 0x{Flasher.Address.ToString("X08")}...");
                Flasher.StepExecute(Flasher.Address);
                SetBtnBackgroundColor(sender, Color.PaleGreen);
            }
            catch (Exception ex) {
                LastResults = $"Sequence ExecuteFirmware->Error:{ex.Message}";
                AddLog("ERROR: " + ex.Message);
                SetBtnBackgroundColor(sender, Color.Salmon);
            }
        }
        #endregion

        void btn_SelectBinary_Click(object sender, EventArgs e) {
            if (DialogResult.OK == openFileDialog1.ShowDialog()) {
                SetFirmwareFile(openFileDialog1.FileName);
            }
        }
    }
}
